/*****************************************************************************/
/*                                                                           */
/*				    error.c				     */
/*                                                                           */
/*		     Error handling functions for backupd		     */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 1998     Ullrich von Bassewitz                                        */
/*              Wacholderweg 14                                              */
/*              D-70597 Stuttgart                                            */
/* EMail:       uz@musoftware.de                                             */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any express or implied         */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <syslog.h>
#include <errno.h>

#include "global.h"
#include "config.h"
#include "error.h"



/*****************************************************************************/
/*     	       	   	  	     Code				     */
/*****************************************************************************/



void warning (const char* format, ...)
/* Log a warning with LOG_WARNING to the syslog */
{
    va_list ap;
    int error = errno;

    va_start (ap, format);
    vsyslog (LOG_WARNING, format, ap);
    va_end (ap);
    errno = error;
}



void error (const char* format, ...)
/* Log an error with LOG_ERR to the syslog */
{
    va_list ap;
    int error = errno;

    va_start (ap, format);
    vsyslog (LOG_ERR, format, ap);
    va_end (ap);
    errno = error;
}



void errexit (const char* format, ...)
/* Log an error with LOG_ERR to the syslog and abort the program */
{
    va_list ap;
    va_start (ap, format);
    vsyslog (LOG_ERR, format, ap);
    va_end (ap);
    exit (EXIT_FAILURE);
}



void usage (void)
/* Print usage information and exit */
{
    fprintf (stderr,
       	     "Usage: %s [options]\n"
       	     "Option list:\n"
       	     "\t-c name\tUse other config file than %s\n"
	     "\t-V\tPrint version information\n",
       	     progname,
  	     configname);
    exit (EXIT_FAILURE);
}



void connbroken (void)
/* Log an error about the broken connection and exit */
{
    warning ("Connection broken");
    exit (EXIT_FAILURE);
}



void opensyslog (void)
/* Open the syslog */
{
    long fac;

    /* Read the default log facility from the logfile */
    CfgGetInt (0, "logfacility", LOG_DAEMON, &fac);

    /* Open the syslog connection */
    openlog (progname, LOG_PID, (int) fac);
}



