/*****************************************************************************/
/*                                                                           */
/*				   config.c				     */
/*                                                                           */
/*		 Read configuration stuff from a config file.		     */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 1998     Ullrich von Bassewitz                                        */
/*              Wacholderweg 14                                              */
/*              D-70597 Stuttgart                                            */
/* EMail:       uz@musoftware.de                                             */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any express or implied         */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <limits.h>
#include <syslog.h>
#include <errno.h>

#include "const.h"
#include "check.h"
#include "util.h"



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



/* Name of the config file to use */
const char* configname = "/etc/backupd.conf";

/* Config file */
static FILE* ConfigFile = 0;



/*****************************************************************************/
/*                             Helper functions                              */
/*****************************************************************************/



static int IsWhite (int C)
/* Test if the character C is white space */
{
    return isascii (C) && isspace (C);
}



static int FindSection (const char* Section)
/* Search forward in the current file until the section with the given name
 * is found. The file pointer is positioned after the line beginning the
 * section. An empty Section is not handled as FAILURE but as a "global"
 * with no header at file start. Returns SUCCESS/FAILURE.
 */
{
    char Line [256];
    int Len = Section? strlen (Section) : 0;

    /* Make sure, the config file is open */
    CHECK (ConfigFile != 0);

    /* Rewind the config file */
    rewind (ConfigFile);

    /* Do we have a section given? */
    if (Len == 0) {
	/* Global section */
     	return SUCCESS;
    }

    /* Section given, search for it */
    while (fgets (Line, sizeof (Line), ConfigFile) != 0) {

	/* Is this a line beginning a section? */
	if (Line [0] != '[') {
	    continue;
	}

	/* Looks like an entry, match the section name */
	if (strncasecmp (Section, Line+1, Len) == 0 && Line [Len+1] == ']') {
	    /* Found */
	    return SUCCESS;
	}

    }

    /* Not found */
    return FAILURE;
}



static int FindEntry (const char* Entry, char* Buf, unsigned BufSize)
/* Search for en entry in the current section. Stop if entry found, EOF or
 * end of section reached. Return SUCCESS or FAILURE.
 */
{
    int EntryLen;
    const char* B;


    /* Make sure, the config file is open */
    CHECK (ConfigFile != 0);

    /* Get length of entry identifer */
    EntryLen = strlen (Entry);

    /* Config file must be open and Entry must have a value */
    if (EntryLen > 0 && BufSize > 0 && Buf != 0) {

        /* Read next line */
        while (fgets (Buf, BufSize, ConfigFile) != 0) {

            /* Is this the start of the next section? */
            if (Buf [0] == '[') {
                /* New section means "not found" */
                return FAILURE;
            }

            /* Skip spaces */
            B = Buf;
            while (IsWhite (*B)) {
                B++;
            }

            /* Is this a comment? */
            if (*B == ';' || *B == '#') {
                /* Comment, proceed with next line */
                continue;
            }

            /* Compare the entry name. If the test succeeds, the next character
             * in the line buffer must be white space or the '=' character.
             */
            if (strncasecmp (B, Entry, EntryLen) == 0) {
                if (IsWhite (B [EntryLen]) || B [EntryLen] == '=') {

                    /* Ok, found. Before returning the buffer, delete the white
                     * space at the beginning and the newline at the end. This will
                     * make life easier for the calling function.
                     */
                    int Len = strlen (B);
                    while (Len > 0 && IsWhite (B [Len-1])) {
                        Len--;
                    }
                    if (Len >= 0) {
                        memmove (Buf, B, Len);
                        Buf [Len] = '\0';
                    } else {
                        Buf [0] = '\0';
                    }

                    /* Found */
                    return SUCCESS;

                }
            }
        }
    }

    /* Not found */
    return FAILURE;
}



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



int CfgGetInt (const char* Section, const char* Entry, long DefVal, long* Val)
/* Read an int from the config file. Val contains the result of the search, the
 * function return value tells if the entry has been found or not (SUCCESS or
 * FAILURE).
 */

{
    char Buf [256];

    /* Don't accept NULL pointers */
    PRECONDITION (Entry != 0 && Val != 0);

    /* Section found or not given. Find the entry */
    if (FindSection (Section) && FindEntry (Entry, Buf, sizeof (Buf))) {

	/* Parse the actual line and return the entry converted to int */
	const char* Pos = strchr (Buf, '=');
	if (Pos) {
	    if (sscanf (Pos+1, "%li", Val) == 1) {
		/* Conversion ok, entry found */
		return SUCCESS;
	    }
	}
    }


    /* Failed */
    *Val = DefVal;
    return FAILURE;
}



int CfgGetStr (const char* Section,
               const char* Entry,
               const char* DefVal,
               char* Str,
               unsigned StrSize)
/* Read a string entry from the config file. Return the default in Str if the
 * section/entry was not found. DefVal and Str may point to the same string!
 * The function itself returns SUCCESS if the entry has been found and FAILURE
 * otherwise.
 */
{
    char Buf [256];

    /* Don't accept NULL pointers */
    PRECONDITION (Entry != 0 && Str != 0);

    /* Find section/entry */
    if (FindSection (Section) && FindEntry (Entry, Buf, sizeof (Buf))) {

	/* A string is delimited by a leading and trailing \" char */
       	char* Rear;
	char* Front = strchr (Buf, '=');
	if (Front) {

	    /* Let Front point behind the '\"' if there is one */
	    Front = strchr (Front, '\"');
	    if (Front) {

		/* Skip the leading delim */
		Front++;

		/* Search for the trailing delimiter */
		Rear = strrchr (Front, '\"');
		if (Rear) {
		    /* We got it! */
		    *Rear = '\0';
		    StrNCopy (Str, Front, StrSize);

       	       	    /* Found */
		    return SUCCESS;
		}

	    }

	}
    }


    /* Failed - return the default value.
     * Beware: DefVal may be the same string as Str.
     */
    if (Str != DefVal) {
	StrNCopy (Str, DefVal, StrSize);
    }
    return FAILURE;
}



int CfgHaveSection (const char* Section)
/* Return YES if the named section exists in the config file, return NO
 * otherwise.
 */
{
    /* Find section/entry */
    return FindSection (Section);
}



unsigned long CfgListSections (FILE* F)
/* List all available sections, each on a line, return the bytes written. */
{
    char Line [256];
    char* Closing;
    unsigned long bytes = 0;

    /* We must have an open config file to do this */
    PRECONDITION (ConfigFile != 0);

    /* Rewind the config file */
    rewind (ConfigFile);

    if (ConfigFile) {

        /* Read line by line */
        while (fgets (Line, sizeof (Line), ConfigFile) != 0) {

            /* Is this a line beginning a section? */
            if (Line [0] != '[') {
                continue;
            }

            /* A section entry */
	    Closing = strrchr (Line, ']');
	    if (Closing == 0) {
		/* OOPS - syntax error. Ignore */
		continue;
	    }

	    /* Setup the line (insert a dash as first char to avoid
	     * interpretation as control command (OK or ERROR).
	     */
	    *Closing = '\0';
	    Line [0] = '-';

	    /* Print it */
	    fprintf (F, "%s\n", Line);

	    /* Count the bytes */
	    bytes += strlen (Line) + 1;

        }
    }

    /* Return the count of bytes written */
    return bytes;

}



int CfgInit (void)
/* Open the config file. Return SUCCESS/FAILURE */
{
    /* Open the config file */
    ConfigFile = fopen (configname, "r");
    if (ConfigFile == 0) {
    	/* Could not open... */
    	syslog (LOG_ERR, "Could not open config file \"%s\": %m", configname);
	return FAILURE;

    } else {

	return SUCCESS;

    }
}



void CfgDone (void)
/* Close the config file if it is open */
{
    if (ConfigFile) {
	fclose (ConfigFile);
	ConfigFile = 0;
    }
}



