/*****************************************************************************/
/*                                                                           */
/*				   backupd.c				     */
/*                                                                           */
/*		       Backup daemon for use with linux.		     */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 1998     Ullrich von Bassewitz                                        */
/*              Wacholderweg 14                                              */
/*              D-70597 Stuttgart                                            */
/* EMail:       uz@musoftware.de                                             */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any express or implied         */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <setjmp.h>
#include <paths.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <sys/socket.h>

#include "const.h"
#include "proto.h"
#include "global.h"
#include "config.h"
#include "error.h"
#include "util.h"
#include "sig.h"
#include "client.h"
#include "extcmd.h"



/*****************************************************************************/
/*	   	  		     data			       	     */
/*****************************************************************************/



/* Last lockfile created (or empty) */
static char lockfile [PATH_MAX] = "";

/* Number of data(!) bytes read and written */
static unsigned long   	wbytes = 0;
static unsigned long    rbytes = 0;

/* Default path */
#ifdef _PATH_STDPATH
#  define PATH	_PATH_STDPATH
#else
#  define PATH  "/usr/bin:/bin:/usr/sbin:/sbin"
#endif



/*****************************************************************************/
/*     	   	 		     code			       	     */
/*****************************************************************************/



static const char* getarg (const char* cmd, unsigned len)
/* Given a command line with command length len, return a pointer to the
 * command argument(s).
 */
{
    /* Skip the command */
    cmd += len;

    /* Skip whitespace */
    while (isascii (*cmd) && isspace (*cmd)) {
	++cmd;
    }

    /* Return the pointer */
    return cmd;
}



static void setup_security (void)
/* Setup security stuff on startup */
{
    char envbuf [256];

    /* Any file created should not be accessible by others */
    umask (0077);

    /* Don't include any unusual paths elements */
    if (snprintf (envbuf, sizeof (envbuf), "PATH=%s", PATH) == -1) {
    	errexit ("Cannot set PATH");
    }
    if (putenv (envbuf) != 0) {
	errexit ("Cannot set PATH");
    }
}



static int translatemeta (char* target, const char* src, unsigned size)
/* Translate src into target expanding meta characters. Return SUCCESS or
 * FAILURE.
 */
{
    int i;
    char* host;

    i = 0;
    while (*src && i < size-1) {

    	if (*src != '\\') {
    	    target [i++] = *src++;
    	    continue;
    	}

    	/* Meta character */
    	switch (*++src) {

    	    case 'h':
    	       	host = clientname;
    	       	while (*host && i < size-1) {
       	       	    target [i++] = *host++;
    	       	}
    	       	++src;
    	       	break;

 	    case 'H':
    	       	host = clientname;
    	       	while (*host && i < size-1) {
	       	    if (validfilechar (*host)) {
	       		/* Use this char from the hostname */
       	       	        target [i++] = *host++;
	       	    } else {
	       		/* Use a dash instead */
	       		target [i++] = '-';
	       	    }
    	       	}
    	       	++src;
    	       	break;

	    case '\0':
	       	/* Must handle this separately */
	       	break;

    	    default:
    	       	target [i++] = *src;
       	       	++src;
    	       	break;

    	}
    }

    /* Terminate the buffer */
    target [i] = '\0';

    /* Check if the copy was successful */
    return (i < size-1);
}



static int getcmd (const char* res, const char* op, char* buf, unsigned size)
/* Read the command for the resource res and operation op into buf. Replace
 * metacharacters. Return SUCCESS or FAILURE;
 */
{
    char cmd [1024];

    /* Read the external command */
    if (CfgGetStr (res, op, "", cmd, sizeof (cmd)) == FAILURE) {
	/* No command given */
	error ("No %s entry for resource \"%s\"", op, res);
	erranswer ("Configuration error");
     	return FAILURE;
    }

    /* Move it into the callers buffer and replace meta chars */
    if (translatemeta (buf, cmd, size) != FAILURE) {
     	/* Log the command */
    	syslog (LOG_INFO, "External cmd is \"%s\"", buf);
 	return SUCCESS;
    } else {
 	error ("Expanding external cmd failed");
 	erranswer ("Internal error");
 	return FAILURE;
    }
}



static void changeuser (const char* section)
/* Change user and group to the values given in the config file */
{
    char user [17];
    char group [17];
    struct passwd* p;
    struct group* g;


    /* Handle group (must do this first) */
    if (CfgGetStr (section, "group", "", group, sizeof (group)) != FAILURE) {
	/* Group was given */
       	g = getgrnam (group);
      	if (g == 0) {
	    errexit ("Group \"%s\" is unknown", group);
	}
	if (setgid (g->gr_gid)< 0) {
       	    errexit ("Cannot change group to \"%s\"", group);
	}
    }

    /* Handle user */
    if (CfgGetStr (section, "user", "", user, sizeof (user)) != FAILURE) {
    	/* User was given */
    	p = getpwnam (user);
	if (p == 0) {
       	    errexit ("User \"%s\" is unknown", user);
	}
	if (setuid (p->pw_uid)< 0) {
       	    errexit ("Cannot change user to \"%s\"", user);
       	}
    }
}



static int makelock (const char* res)
/* Make a lock if the configuration requests one. Returns SUCCESS/FAILURE */
{
    int fd;
    char name [PATH_MAX];

    /* Read the name of the lockfile from the configuration */
    if (CfgGetStr (res, "lockfile", "", name, sizeof (name)) == FAILURE) {
       	/* No need for a lock */
       	return SUCCESS;
    }

    /* Expand the lockfile name */
    if (translatemeta (lockfile, name, sizeof (lockfile)) == FAILURE) {
       	/* Expand failed */
       	error ("Could not expand lock file name");
       	return FAILURE;
    }

    /* Create the lockfile */
    if ((fd = open (lockfile, O_CREAT | O_EXCL, 0600)) == -1) {
       	/* Could not create the lock. Be shure to clear the name */
       	lockfile [0] = '\0';
       	return FAILURE;
    }

    /* Success - close the file */
    close (fd);
    return SUCCESS;
}



static void rmlock (void)
/* Remove the last created lockfile */
{
    if (strlen (lockfile) > 0) {
	if (remove (lockfile) < 0) {
       	    error ("Cannot remove lock file \"%s\": %m", lockfile);
	}
	lockfile [0] = '\0';
    }
}



static void doexit (void)
/* Exit function, will remove existing lockfiles */
{
    rmlock ();
}



static int readwrite (const char* res, const char* op, char* cmd, unsigned size)
/* Common stuff for doread and dowrite. Check the argument and the config
 * file. Read the external command into cmd. Return SUCCESS or FAILURE.
 */
{
    long mask;


    /* Check the argument list */
    if (strlen (res) == 0) {
    	warning ("%s: Missing argument", op);
    	erranswer ("Missing argument");
    	return FAILURE;
    }

    /* Check if we have the config section given as an argument */
    if (CfgHaveSection (res) == NO) {
    	/* Trying to access an unknown resource */
    	warning ("%s: Unknown resource: \"%s\"", op, res);
    	erranswer ("Unknown resource: %s", res);
    	return FAILURE;
    }

    /* Check if the client has access to this resource */
    if (clientaccess (res, "allow", "deny") == NO) {
	error ("Access to resource \"%s\" denied", res);
	erranswer ("Access denied");
    	return FAILURE;
    }

    /* Read user/group from the config file and change identity */
    changeuser (res);

    /* Change the umask value if one is given */
    if (CfgGetInt (res, "umask", 0022, &mask) != FAILURE) {
    	umask ((int) mask);
    }

    /* Create a lock file if needed */
    if (makelock (res) == FAILURE) {
	warning ("%s: Cannot make lock for resource \"%s\"", op, res);
	erranswer ("Resource is locked");
	return FAILURE;
    }

    /* Read and expand the external command */
    return getcmd (res, op, cmd, size);
}



static void closepipe (void)
/* Close the command pipe and send a status message to the client */
{
    /* Close the pipe and get the client return code. */
    int rc = endcmd ();
    if (rc == -1) {
	/* Could not execute */
	int err = errno;
	error ("Error closing pipe: %m");
	erranswer ("%s", strerror (err));
	/* Do not terminate here */
    } else if ((rc & 0x7F00) == 0x7F00) {
	error ("Could not execute external command");
	erranswer ("Could not execute external command");
    } else if ((rc & 0x0080) != 0) {
	warning ("External command terminated by signal %d", rc & 0x7F);
       	erranswer ("External command terminated by signal %d", rc & 0x7F);
    } else if (rc != 0) {
	/* This is the exit code of the child */
	warning ("External command exited with code %d", rc);
	erranswer ("External command exited with code %d", rc);
    } else {
	/* Ok */
	okanswer ();
    }
}



static void dowrite (const char* res)
/* Handle the WRITE command */
{
    char buf [256];
    char cmd [1024];

    /* Do some checks */
    if (readwrite (res, CMD_WRITE, cmd, sizeof (cmd)) != SUCCESS) {
	return;
    }

    /* Start the external command and connect it's input to clientio */
    startcmd (cmd, "w");

    /* Send an ok */
    okanswer ();

    /* Receive data */
    while (1) {

	/* Get char count in next block */
       	int count = getc (stdin);
	if (count == EOF) {
	    connbroken ();
	}

	/* Zero sized block means end of data */
	if (count == 0) {
	    break;
	}

       	/* Read/write the next block of data */
	while (count) {
       	    int c = fread (buf, 1, count, stdin);
	    if (c == 0) {
	       	connbroken ();
	    }
	    fwrite (buf, 1, c, clientio);
	    count -= c;
	    wbytes += c;
      	}
    }

    /* Close the pipe */
    closepipe ();

    /* Remove the lockfile */
    rmlock ();
}



static void doread (const char* res)
/* Handle the READ command */
{
    char buf [256];
    char cmd [1024];

    /* Do some checks */
    if (readwrite (res, CMD_READ, cmd, sizeof (cmd)) != SUCCESS) {
	return;
    }

    /* Start the external command and connect it's output to clientio */
    startcmd (cmd, "r");

    /* Send an ok */
    okanswer ();

    /* Send data */
    while (1) {

	/* Read next block */
	int count = fread (buf, 1, 255, clientio);
	if (count == 0) {
	    break;
	}

	/* Write block */
	putc (count, stdout);
       	if (fwrite (buf, 1, count, stdout) != count) {
	    connbroken ();
	}

       	/* Keep track of the bytes read */
	rbytes += count;
    }

    /* Send the end marker. Don't check for errors since we will close
     * the pipe anyway.
     */
    putc (0, stdout);
    fflush (stdout);

    /* Close the pipe */
    closepipe ();

    /* Remove the lockfile */
    rmlock ();
}



static void dolist (const char* args)
/* List all available resources */
{
    /* Is the client allowed to use the LIST command? */
    if (clientaccess (0, "listallow", "listdeny") == NO) {
    	erranswer ("Access denied");
    } else {
	/* This one is simple... */
	wbytes += CfgListSections (stdout);
	okanswer ();
    }
}



int main (int argc, char* argv [])
{
    int        	  opt;        	   	/* Current cmd line option */
    char      	  cmd [256];		/* Client input buffer */
    time_t     	  starttime;		/* Remember the start time */
    unsigned long connecttime;		/* Duration of the connection */


    /* First thing: Ignore SIGPIPE so we won't terminate if the client
     * disconnects. As we're in this thing here, ignore some others, too.
     */
    sigignore (SIGPIPE);
    sigignore (SIGHUP);
    sigignore (SIGINT);
    sigignore (SIGQUIT);
    sigignore (SIGUSR1);
    sigignore (SIGUSR2);

    /* Initialize variables */
    progname = argv [0];

    /* Activate our cleanup function */
    atexit (doexit);

    /* Remember the startup time */
    starttime = time (0);

    /* Parse program options */
    while ((opt = getopt (argc, argv, "c:")) != EOF) {

       	switch (opt) {

       	    case 'c':
       	      	configname = optarg;
       	      	break;

	    case 'V':
		fprintf (stderr, "Version: %s\n", VERSION);
		break;

       	    case '?':
       	      	usage ();
       	      	break;

     	}
    }

    /* Open the config file. We have a race condition here, since we want to
     * read the log facility from the config file, so if opening the file
     * fails, we will write to the syslog without proper setup. Since this
     * situation is not very common, ignore it.
     */
    if (CfgInit () != SUCCESS) {
	/* OOPS */
	errexit ("Error accessing config file \"%s\"", configname);
    }

    /* Setup security stuff */
    setup_security ();

    /* Open the syslog connection */
    opensyslog ();

    /* Get the client name and log it. */
    getclient ();
    syslog (LOG_NOTICE, "Connect from %s", clientname);

    /* Set up a long jump label that is used if we catch a signal. This is
     * currently only needed to remove lockfile if we get a signal that would
     * other cause immediate program termination. The stuff above is not
     * critical, so it's done here, as soon as we begin to talk to the
     * client.
     */
    if (sigsetjmp (sigexit, 1) == 0) {

      	/* Now, as we've setup our label, catch some signals. Linux generates
	 * a SIGBUS if memory is low - catch that too.
	 */
        sigset (SIGTERM, sighandler);
      	sigset (SIGPWR,  sighandler);
	sigset (SIGBUS,  sighandler);

       	/* Read incoming commands */
       	while (fgets (cmd, sizeof (cmd), stdin) != 0) {

       	    /* Remove trailing white space including the newline */
       	    int len = strlen (cmd);
       	    while (len > 0 && isascii (cmd [len-1]) && isspace (cmd [len-1])) {
       		cmd [--len] = '\0';
       	    }

       	    /* For convenience and debugging, ignore empty commands */
       	    if (len == 0) {
       		continue;
       	    }

       	    /* Log the command we got */
       	    syslog (LOG_INFO, "Remote command: \"%s\"", cmd);

       	    /* The first part of the received line is expected to be a command. */
       	    if (strncmp (cmd, CMD_WRITE, strlen (CMD_WRITE)) == 0) {

       		/* Handle the WRITE command */
		dowrite (getarg (cmd, strlen (CMD_WRITE)));

	    } else if (strncmp (cmd, CMD_READ, strlen (CMD_READ)) == 0) {

		/* Handle the READ command */
		doread (getarg (cmd, strlen (CMD_READ)));

	    } else if (strncmp (cmd, CMD_LIST, strlen (CMD_LIST)) == 0) {

		/* Handle the LIST command */
		dolist (getarg (cmd, strlen (CMD_LIST)));

	    } else if (strncmp (cmd, CMD_BYE, strlen (CMD_BYE)) == 0) {

		/* Done */
		break;

	    } else {

		/* Unknown command */
		warning ("Unknown command: \"%s\"", cmd);
		erranswer ("Unknown command");

	    }

	}

    } else {

	/* We got a signal. Write something to the logfile and terminate */
       	error ("Caught signal %d - will terminate", sigcaught);

    }

    /* Close the config file */
    CfgDone ();

    /* Calculate the connection time in seconds */
    connecttime = (unsigned long) difftime (time (0), starttime);

    /* Done */
    syslog (LOG_NOTICE,
       	    "Terminated. Read: %lu Write: %lu Time: %lu",
	    rbytes,
	    wbytes,
	    connecttime);
    return EXIT_SUCCESS;
}




