/*****************************************************************************/
/*                                                                           */
/*				    util.c				     */
/*                                                                           */
/*			      Utility functions.			     */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 1998     Ullrich von Bassewitz                                        */
/*              Wacholderweg 14                                              */
/*              D-70597 Stuttgart                                            */
/* EMail:       uz@musoftware.de                                             */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any express or implied         */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <limits.h>
#include <unistd.h>
#include <sys/stat.h>

#include "const.h"
#include "util.h"



/*****************************************************************************/
/*				     Data				     */
/*****************************************************************************/



/* This was originally in spunk */
static const char FileSysPathSep = '/';



/*****************************************************************************/
/*				     Code				     */
/*****************************************************************************/



char* AddPathSep (char* Dir)
/* Add a trailing path separator if the given path does not end in one */
{
    unsigned Len = strlen (Dir);
    if (Len == 0 || Dir [Len-1] != FileSysPathSep) {
	Dir [Len]   = FileSysPathSep;
	Dir [Len+1] = '\0';
    }
    return Dir;
}



char* DelPathSep (char* Path)
/* Delete a trailing path separator if the last char of path is one */
{
    unsigned Len = strlen (Path);
    while (Len > 0 && Path [Len-1] == FileSysPathSep) {
	Path [--Len] = '\0';
    }
    return Path;
}



char* DirCat (char* Path, const char* NewElement)
/* Add the new path element in NewElement to the given path. The function makes
 * shure that the path separator is handled correct. The complete path is
 * stored in Path, a pointer to path is returned.
 */
{
    /* Add a trailing path separator if Path does not have one */
    AddPathSep (Path);

    /* Skip leading path separators in NewElement */
    while (*NewElement == FileSysPathSep) {
	NewElement++;
    }

    /* Concatenate the strings and return the result */
    return strcat (Path, NewElement);
}



int IsDir (const char* Path)
/* Return YES if the name denotes an existing directory, return NO if not */
{
    struct stat Buf;

    /* Copy the path to a temp location, so we can remove an eventually
     * existing path separator.
     */
    char TmpBuf [PATH_MAX+1];
    DelPathSep (StrNCopy (TmpBuf, Path, sizeof (TmpBuf)));

    /* Now try to stat the file */
    return stat (TmpBuf, &Buf) == 0 && S_ISDIR (Buf.st_mode);
}



char* StrNCopy (char* Dest, const char* Src, size_t Size)
/* Does what strncpy should do (but doesn't): Copy Src to Dest, where Size is
 * the Size of the memory location that Dest points to. The resulting string
 * is terminated with a NUL character in any case. The function returns a
 * pointer to Dest.
 */
{
    assert (Dest != 0 && Src != 0 && Size > 0);

    strncpy (Dest, Src, Size-1);
    Dest [Size-1] = '\0';

    return Dest;
}



int IsReadable (const char* Path)
/* Return YES if the file with the given name exists and is readable,
 * return NO if not.
 */
{
    return access (Path, R_OK) == 0;
}



int IsAbsolute (const char* Path)
/* Return YES is Path is an absolute path. Return NO if not. */
{
    int Len = strlen (Path);

    /* An empty path amy not be absolute */
    if (Len == 0) {
	return NO;
    }

    /* Beware: There may be a leading drive spec */
    if (Len > 2 && Path [1] == ':') {
	/* look behind the drive specifier */
	return Path [2] == '\\';
    } else {
	/* Look at the first character of the path name */
	return Path [0] == '\\';
    }
}



