/*****************************************************************************/
/*                                                                           */
/*				   config.c				     */
/*                                                                           */
/*		 Read configuration stuff from a config file.		     */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 1998     Ullrich von Bassewitz                                        */
/*              Wacholderweg 14                                              */
/*              D-70597 Stuttgart                                            */
/* EMail:       uz@musoftware.de                                             */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any express or implied         */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <limits.h>
#include <syslog.h>
#include <errno.h>

#include "const.h"
#include "util.h"



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



/* Config file */
static FILE* ConfigFile = 0;



/*****************************************************************************/
/*                             Helper functions                              */
/*****************************************************************************/



static int IsWhite (int C)
/* Test if the character C is white space */
{
    return isascii (C) && isspace (C);
}



static int FindSection (const char* Section)
/* Search forward in the current file until the section with the given name
 * is found. The file pointer is positioned after the line beginning the
 * section. Returns SUCCESS/FAILURE.
 * If no config file is open, the function returns FAILURE.
 */
{
    char Line [256];
    int Len = strlen (Section);

    if (ConfigFile != 0 && Len > 0) {

        /* Read line by line */
        while (fgets (Line, sizeof (Line), ConfigFile) != 0) {

            /* Is this a line beginning a section? */
            if (Line [0] != '[') {
                continue;
            }

            /* Looks like an entry, match the section name */
            if (strncasecmp (Section, Line+1, Len) == 0 && Line [Len+1] == ']') {
                /* Found */
                return SUCCESS;
            }

        }

    }

    /* Not found */
    return FAILURE;
}



static int FindEntry (const char* Entry, char* Buf, unsigned BufSize)
/* Search for en entry in the current section. Stop if entry found, EOF or
 * end of section reached. Return SUCCESS or FAILURE.
 */
{
    int EntryLen;
    const char* B;


    /* Get length of entry identifer */
    EntryLen = strlen (Entry);

    /* Config file must be open and Entry must have a value */
    if (ConfigFile != 0 && EntryLen > 0 && BufSize > 0 && Buf != 0) {

        /* Read next line */
        while (fgets (Buf, BufSize, ConfigFile) != 0) {

            /* Is this the start of the next section? */
            if (Buf [0] == '[') {
                /* New section means "not found" */
                return FAILURE;
            }

            /* Skip spaces */
            B = Buf;
            while (IsWhite (*B)) {
                B++;
            }

            /* Is this a comment? */
            if (*B == ';' || *B == '#') {
                /* Comment, proceed with next line */
                continue;
            }

            /* Compare the entry name. If the test succeeds, the next character
             * in the line buffer must be white space or the '=' character.
             */
            if (strncasecmp (B, Entry, EntryLen) == 0) {
                if (IsWhite (B [EntryLen]) || B [EntryLen] == '=') {

                    /* Ok, found. Before returning the buffer, delete the white
                     * space at the beginning and the newline at the end. This will
                     * make life easier for the calling function.
                     */
                    int Len = strlen (B);
                    while (Len > 0 && IsWhite (B [Len-1])) {
                        Len--;
                    }
                    if (Len >= 0) {
                        memmove (Buf, B, Len);
                        Buf [Len] = '\0';
                    } else {
                        Buf [0] = '\0';
                    }

                    /* Found */
                    return SUCCESS;

                }
            }
        }
    }

    /* Not found */
    return FAILURE;
}



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



long CfgGetInt (const char* Section, const char* Entry, long DefVal)
/* Read an int from the config file. Return the default if config file not
 * open or section/entry not found.
 */
{
    char Buf [256];

    /* Rewind the config file. */
    rewind (ConfigFile);

    /* Find section/entry */
    if (FindSection (Section) && FindEntry (Entry, Buf, sizeof (Buf))) {

	/* Parse the actual line and return the entry converted to int */
	const char* Pos = strchr (Buf, '=');
	if (Pos) {
	    long Value;
	    if (sscanf (Pos+1, "%li", &Value) == 1) {			 
		/* Return the result */
		return Value;
	    }
	}
    }


    /* Failed - return the default value */
    return DefVal;
}



char* CfgGetStr (const char* Section,
                 const char* Entry,
                 const char* DefVal,
                 char* Str,
                 unsigned StrSize)
/* Read a string entry from the config file. Return the default if config file
 * not open or section/entry not found.
 * DefVal and Str may point to the same string!
 */
{
    char Buf [256];

    /* Rewind the config file. */
    rewind (ConfigFile);

    /* Find section/entry */
    if (FindSection (Section) && FindEntry (Entry, Buf, sizeof (Buf))) {

	/* A string is delimited by a leading and trailing \" char */
	const char* Rear;
	const char* Front = strchr (Buf, '=');
	if (Front) {

	    /* Let Front point behind the '\"' if there is one */
	    Front = strchr (Front, '\"');
	    if (Front) {

		/* Skip the leading delim */
		Front++;

		/* Search for the trailing delimiter */
		Rear = strrchr (Front, '\"');
		if (Rear) {
		    /* OOPS! We got it! */
		    unsigned Len = Rear - Front;
		    if (Len > StrSize-1) {
			Len = StrSize-1;
		    }
		    memcpy (Str, Front, Len);
		    Str [Len] = '\0';

       	       	    /* Return the result */
		    return Str;
		}

	    }

	}
    }


    /* Failed - return the default value.
     * Beware: DefVal may be the same string as Str.
     */
    if (Str != DefVal) {
        strncpy (Str, DefVal, StrSize-1);
        Str [StrSize] = '\0';
    }
    return Str;
}



char* CfgGetDir (const char* Section,
		 const char* Entry,
		 const char* DefVal,
		 char* Dir)
/* Use CfgGetStr to read in a string. Interpret this string as a directory,
 * clean it up by removing a trailing path separator and make it an absolute
 * path. Dir is assumed to point to at least _MAX_PATH bytes.
 */
{
    /* Read the entry from the file */
    CfgGetStr (Section, Entry, DefVal, Dir, PATH_MAX);

    /* Remove a trailing path separator if needed */
    DelPathSep (Dir);

    /* Return the resulting string */
    return Dir;
}



int CfgHaveSection (const char* Section)
/* Return YES if the named section exists in the config file, return NO
 * otherwise.
 */
{
    /* Rewind the config file */
    rewind (ConfigFile);

    /* Find section/entry */
    return FindSection (Section);
}



int CfgInit (const char* FileName)
/* Open the config file. Return SUCCESS/FAILURE */
{
    /* Open the config file */
    ConfigFile = fopen (FileName, "r");
    if (ConfigFile == 0) {
    	/* Could not open... */
    	syslog (LOG_ERR,
    		"Could not open config file \"%s\": %s",
		FileName,
    		strerror (errno));
	return FAILURE;

    } else {

	return SUCCESS;

    }
}



void CfgDone (void)
/* Close the config file if it is open */
{
    if (ConfigFile) {
	fclose (ConfigFile);
	ConfigFile = 0;
    }
}



