/*****************************************************************************/
/*                                                                           */
/*				   backupd.c				     */
/*                                                                           */
/*		       Backup daemon for use with linux.		     */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 1998     Ullrich von Bassewitz                                        */
/*              Wacholderweg 14                                              */
/*              D-70597 Stuttgart                                            */
/* EMail:       uz@musoftware.de                                             */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any express or implied         */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <netdb.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "const.h"
#include "config.h"



/*****************************************************************************/
/*	   	 		     data			       	     */
/*****************************************************************************/



/* Command strings */
#define CMD_WRITE    	"WRITE"
#define CMD_READ	"READ"
#define CMD_BYE	     	"BYE"

/* The escape character used when transfering data */
#define DESC		'u'

/* Control character to end a data connection */
#define DEND		'e'

/* Name of the program (set to argv [0]) */
static const char* progname = "backupd";

/* Log facility */
static unsigned logfac	= LOG_DAEMON;

/* Name of the config file */
static const char* configname = "/etc/backupd.conf";

/* Last lockfile created (or empty) */
static char lockfile [PATH_MAX] = "";

/* IP address of the client in host byte order */
static unsigned long clientaddr = 0x7F000001;	/* 127.0.0.1 */

/* Name or IP of the client in readable form */
static char clientname [1024] = "";

/* Number of data(!) bytes read and written */
static unsigned long   	wbytes = 0;
static unsigned long    rbytes = 0;



/*****************************************************************************/
/*     	   	 		     code			       	     */
/*****************************************************************************/



static void usage (void)
/* Print usage information and exit */
{
    fprintf (stderr,
       	     "Usage: %s [options]\n"
       	     "Option list:\n"
       	     "\t-c name\tUse other config file than %s\n"
       	     "\t-l n\tUse logging facility n\n",
       	     progname,
	     configname);
    exit (EXIT_FAILURE);
}



static void connbroken (void)
/* Log an error and exit */
{
    syslog (LOG_WARNING, "Connection broken");
    exit (EXIT_FAILURE);
}



static int validfilechar (int c)
/* Return YES if the given character is valid for filenames, return NO if
 * not. Be very conservative here...
 */
{
    return (isascii (c) &&
	    (isalpha (c) || isdigit (c) || c == '-' || c == '.'));
}



static const char* getarg (const char* cmd, unsigned len)
/* Given a command line with command length len, return a pointer to the
 * command argument(s).
 */
{
    /* Skip the command */
    cmd += len;

    /* Skip whitespace */
    while (isascii (*cmd) && isspace (*cmd)) {
	++cmd;
    }

    /* Return the pointer */
    return cmd;
}



static int translatemeta (char* target, const char* src, unsigned size)
/* Translate src into target expanding meta characters. Return SUCCESS or
 * FAILURE.
 */
{
    int i;
    char* host;

    i = 0;
    while (*src && i < size-1) {

    	if (*src != '\\') {
    	    target [i++] = *src++;
    	    continue;
    	}

    	/* Meta character */
    	switch (*++src) {

    	    case 'h':
    	       	host = clientname;
    	       	while (*host && i < size-1) {
       	       	    target [i++] = *host++;
    	       	}
    	       	++src;
    	       	break;

 	    case 'H':
    	       	host = clientname;
    	       	while (*host && i < size-1) {
	       	    if (validfilechar (*host)) {
	       		/* Use this char from the hostname */
       	       	        target [i++] = *host++;
	       	    } else {
	       		/* Use a dash instead */
	       		target [i++] = '-';
	       	    }
    	       	}
    	       	++src;
    	       	break;

	    case '\0':
	       	/* Must handle this separately */
	       	break;

    	    default:
    	       	target [i++] = *src;
       	       	++src;
    	       	break;

    	}
    }

    /* Terminate the buffer */
    target [i] = '\0';

    /* Check if the copy was successful */
    return (i < size-1);
}



static void getclient (void)
/* Determine the name of the client. Abort on errors. */
{
    struct hostent* h;
    struct sockaddr_in client;
    int len = sizeof (client);
    char* c;

    /* Determine the IP address */
    if (getpeername (fileno (stdin), (struct sockaddr*) &client, &len) == -1) {
       	/* Some sort of error. To allow testing without having an IP
	 * connection, check if stdin is not a socket, if so assume stdin
	 * and do not abort.
	 */
       	syslog (LOG_ERR, "Cannot determine client IP address (%d)", errno);
	if (errno != ENOTSOCK) {
    	    exit (EXIT_FAILURE);
	} else {
	    /* Stdin is not a socket */
	    strcpy (clientname, "stdin");
	    return;
	}
    }

    /* Remember the IP of the client */
    clientaddr = ntohl (client.sin_addr.s_addr);

    /* Try to map the address into a name */
    h = gethostbyaddr ((const char*)&client.sin_addr.s_addr,
	      	       sizeof (client.sin_addr.s_addr),
	      	       AF_INET);
    if (h == 0) {

 	/* Log the error */
 	if (h_errno == -1) {
 	    syslog (LOG_WARNING,
 	      	    "Cannot resolve client address: %s(%d)",
 	      	    strerror (errno),
 	      	    errno);
 	} else {
	    char* msg;
	    switch (h_errno) {
		case HOST_NOT_FOUND:
		    msg = "host not found";
		    break;

		case NO_ADDRESS:
		    msg = "no IP address for host";
		    break;

		case NO_RECOVERY:
		    msg = "unrecoverable DNS error";
		    break;

		case TRY_AGAIN:
		    msg = "temporary DNS error (try again)";
		    break;

		default:
		    msg = "unknown return code";
		    break;
	    }
 	    syslog (LOG_WARNING,
 	      	    "Cannot resolve client address: %s(%d)",
 	      	    msg,
 	      	    h_errno);
 	}

 	/* Use the address as name */
 	sprintf (clientname,
 		 "%u.%u.%u.%u",
       	       	 (unsigned) (clientaddr >> 24) & 0xFF,
       	       	 (unsigned) (clientaddr >> 16) & 0xFF,
       	       	 (unsigned) (clientaddr >> 8) & 0xFF,
       	       	 (unsigned) (clientaddr >> 0) & 0xFF);

    } else {

 	/* We have a name */
       	strncpy (clientname, h->h_name, sizeof (clientname)-1);
 	clientname [sizeof (clientname)-1] = '\0';

    }

    /* Security check: Don't accept whitespace here (should
     * never ever happen).
     */
    c = clientname;
    while (*c) {
       	if (isspace (*c++)) {
     	    syslog (LOG_CRIT, "Whitespace in client name!");
     	    exit (EXIT_FAILURE);
     	}
    }
}



static void answer (const char* format, ...)
/* Send an answer to the client, check for errors */
{
    char buf [1024];
    va_list ap;

    /* Create the output line */
    va_start (ap, format);
    vsnprintf (buf, sizeof (buf), format, ap);
    va_end (ap);

    /* Write the output line */
    if (puts (buf) == EOF) {
	/* Error writing stdout - connection broken */
	connbroken ();
    }

    /* Flush the output */
    fflush (stdout);
}



static int getcmd (const char* dev, const char* op, char* buf, unsigned size)
/* Read the command for the resource dev and operation op into buf. Replace
 * metacharacters. Return SUCCESS or FAILURE;
 */
{
    char cmd [1024];

    /* Read the external command */
    CfgGetStr (dev, op, "", cmd, sizeof (cmd));
    if (strlen (cmd) == 0) {
	/* No command given */
	syslog (LOG_ERR, "No %s entry for resource \"%s\"", op, dev);
	answer ("ERROR: Configuration error");
     	return FAILURE;
    }

    /* Move it into the callers buffer and replace meta chars */
    if (translatemeta (buf, cmd, size) != FAILURE) {
    	/* Log the command */
    	syslog (LOG_INFO, "External cmd is \"%s\"", buf);
 	return SUCCESS;
    } else {
 	syslog (LOG_ERR, "Expanding external cmd failed");
 	answer ("ERROR: Internal error");
 	return FAILURE;
    }
}



static void ignoresig (int sig)
/* Ignore the given signal */
{
    struct sigaction action, oldaction;
    action.sa_handler = SIG_IGN;
    sigemptyset (&action.sa_mask);
    action.sa_flags   = SA_RESTART;    	/* Restart interrupted system calls */
    if (sigaction (sig, &action, &oldaction) != 0) {
       	syslog (LOG_ERR, "Error ignoring signal %d: %s", sig, strerror (errno));
	exit (EXIT_FAILURE);
    }
}



static void changeuser (const char* section)
/* Change user and group to the values given in the config file */
{
    char user [17];
    char group [17];
    struct passwd* p;
    struct group* g;

    /* Read the config file values */
    CfgGetStr (section, "user", "", user, sizeof (user));
    CfgGetStr (section, "group", "", group, sizeof (group));

    /* Handle group (must do this first) */
    if (strlen (group) > 0) {
	/* Group was given */
       	g = getgrnam (group);
      	if (g == 0) {
	    syslog (LOG_ERR, "Group \"%s\" is unknown", group);
	    exit (EXIT_FAILURE);
	}
	if (setgid (g->gr_gid)< 0) {
	    syslog (LOG_ERR, "Cannot change group to \"%s\"", group);
       	    exit (EXIT_FAILURE);
	}
    }

    /* Handle user */
    if (strlen (user) > 0) {
    	/* User was given */
    	p = getpwnam (user);
	if (p == 0) {
	    syslog (LOG_ERR, "User \"%s\" is unknown", user);
	    exit (EXIT_FAILURE);
	}
	if (setuid (p->pw_uid)< 0) {
	    syslog (LOG_ERR, "Cannot change user to \"%s\"", user);
	    exit (EXIT_FAILURE);
	}
    }
}



static int makelock (const char* dev)
/* Make a lock if the configuration requests one. Returns SUCCESS/FAILURE */
{
    int fd;
    char name [PATH_MAX];

    /* Read the name of the lockfile from the configuration */
    CfgGetStr (dev, "lockfile", "", name, sizeof (name));

    /* Do we have a lockfile name given? */
    if (strlen (name) == 0) {
       	/* No need for a lock */
       	return SUCCESS;
    }

    /* Expand the lockfile name */
    if (translatemeta (lockfile, name, sizeof (lockfile)) == FAILURE) {
	/* Expand failed */
	syslog (LOG_ERR, "Could not expand lock file name");
	return FAILURE;
    }

    /* Create the lockfile */
    if ((fd = open (lockfile, O_CREAT | O_EXCL, 0600)) == -1) {
       	/* Could not create the lock. Be shure to clear the name */
	lockfile [0] = '\0';
       	return FAILURE;
    }

    /* Success - close the file */
    close (fd);
    return SUCCESS;
}



static void rmlock (void)
/* Remove the last created lockfile */
{
    if (strlen (lockfile) > 0) {
	if (remove (lockfile) < 0) {
	    syslog (LOG_ERR,
	 	    "Cannot remove lock file \"%s\": %s",
	 	    lockfile,
		    strerror (errno));
	}
	lockfile [0] = '\0';
    }
}



static void doexit (void)
/* Exit function, will remove existing lockfiles */
{
    rmlock ();
}



static int readwrite (const char* dev, const char* op, char* cmd, unsigned size)
/* Common stuff for doread and dowrite. Check the argument and the config
 * file. Read the external command into cmd. Return SUCCESS or FAILURE.
 */
{
    int mask;


    /* Check the argument list */
    if (strlen (dev) == 0) {
    	syslog (LOG_WARNING, "%s: Missing argument", op);
    	answer ("ERROR: Missing argument");
    	return FAILURE;
    }

    /* Check if we have the config section given as an argument */
    if (CfgHaveSection (dev) == NO) {
    	/* Trying to access an unknown resource */
    	syslog (LOG_WARNING, "%s: Unknown resource: \"%s\"", op, dev);
    	answer ("ERROR: Unknown resource");
    	return FAILURE;
    }

    /* Read user/group from the config file and change identity */
    changeuser (dev);

    /* Change the umask value if one is given */
    mask = CfgGetInt (dev, "umask", 0022);
    umask (mask);

    /* Create a lock file if needed */
    if (makelock (dev) == FAILURE) {
	syslog (LOG_WARNING, "%s: Cannot make lock for resource \"%s\"", op, dev);
	answer ("ERROR: Resource is locked");
	return FAILURE;
    }

    /* Read and expand the external command */
    return getcmd (dev, op, cmd, size);
}



static void closepipe (FILE* F)
/* Close the command pipe and send a status message to the client */
{
    int rc;

    /* Close the pipe */
    rc = pclose (F);
    if (rc == -1) {
	/* Could not execute */
	int err = errno;
	syslog (LOG_ERR, "Error closing pipe: %s", strerror (err));
	answer ("ERROR: %s", strerror (err));
	/* Do not terminate here */
    } else if ((rc & 0x7F00) == 0x7F00) {
	syslog (LOG_ERR, "Could not execute external command");
	answer ("ERROR: Could not execute external command");
    } else if ((rc & 0x0080) != 0) {
	syslog (LOG_WARNING, "External command terminated by signal %d", rc & 0x7F);
       	answer ("ERROR: External command terminated by signal %d", rc & 0x7F);
    } else if (rc != 0) {
	/* This is the exit code of the child */
	syslog (LOG_WARNING, "External command exited with code %d", rc);
	answer ("ERROR: External command exited with code %d", rc);
    } else {
	/* Ok */
	answer ("OK");
    }
}



static void dowrite (const char* dev)
/* Handle the WRITE command */
{
    char buf [256];
    char cmd [1024];
    FILE* F;

    /* Do some checks */
    if (readwrite (dev, CMD_WRITE, cmd, sizeof (cmd)) != SUCCESS) {
	return;
    }

    /* Create a pipe and connect it to the external command */
    F = popen (cmd, "w");
    if (F == 0) {
	syslog (LOG_ERR, "Cannot open pipe: %s", strerror (errno));
	answer ("ERROR:	Cannot create child process");
	return;
    }

    /* Send an ok */
    answer ("OK");

    /* Receive data */
    while (1) {

	/* Get char count in next block */
       	int count = getc (stdin);
	if (count == EOF) {
	    connbroken ();
	}

	/* Zero sized block means end of data */
	if (count == 0) {
	    break;
	}

	/* Read/write the next block of data */
	while (count) {
       	    int c = fread (buf, 1, count, stdin);
	    if (c == 0) {
	 	connbroken ();
	    }
	    fwrite (buf, 1, c, F);
	    count -= c;
	    wbytes += c;
      	}
    }

    /* Close the pipe */
    closepipe (F);

    /* Remove the lockfile */
    rmlock ();
}



static void doread (const char* dev)
/* Handle the READ command */
{
    char buf [256];
    char cmd [1024];
    FILE* F;

    /* Do some checks */
    if (readwrite (dev, CMD_READ, cmd, sizeof (cmd)) != SUCCESS) {
	return;
    }

    /* Create a pipe and connect it to the external command */
    F = popen (cmd, "r");
    if (F == 0) {
	syslog (LOG_ERR, "Cannot open pipe: %s", strerror (errno));
	answer ("ERROR:	Cannot create child process");
	return;
    }

    /* Send an ok */
    answer ("OK");

    /* Send data */
    while (1) {

	/* Read next block */
	int count = fread (buf, 1, 255, F);
	if (count == 0) {
	    break;
	}

	/* Write block */
	putc (count, stdout);
       	if (fwrite (buf, 1, count, stdout) != count) {
	    connbroken ();
	}

       	/* Keep track of the bytes read */
	rbytes += count;
    }

    /* Send the end marker. Don't check for errors since we will close
     * the pipe anyway.
     */
    putc (0, stdout);
    fflush (stdout);

    /* Close the pipe */
    closepipe (F);

    /* Remove the lockfile */
    rmlock ();
}



int main (int argc, char* argv [])
{
    int opt;   	   		/* Current cmd line option */
    char cmd [256];		/* Client input buffer */

    /* Initialize variables */
    progname = argv [0];

    /* Activate our cleanup function */
    atexit (doexit);

    /* Ignore some signals */
    ignoresig (SIGPIPE);

    /* Parse program options */
    while ((opt = getopt (argc, argv, "c:l:")) != EOF) {

	switch (opt) {

	    case 'c':
		configname = optarg;
		break;

	    case 'l':
		logfac = atoi (optarg);
		break;

	    case '?':
		usage ();
    		break;

     	}
    }

    /* Open the syslog connection */
    openlog (progname, LOG_PID, logfac);

    /* Get the client name and log it. */
    getclient ();
    syslog (LOG_NOTICE, "Connection from %s", clientname);

    /* Open the config file */
    if (CfgInit (configname) != SUCCESS) {
	/* OOPS */
	syslog (LOG_ERR, "Error accessing config file \"%s\"", configname);
	exit (EXIT_FAILURE);
    }

    /* Read incoming commands */
    while (fgets (cmd, sizeof (cmd), stdin) != 0) {

	/* Remove trailing white space including the newline */
	int len = strlen (cmd);
	while (len > 0 && isascii (cmd [len-1]) && isspace (cmd [len-1])) {
	    cmd [--len] = '\0';
	}

	/* For convenience and debugging, ignore empty commands */
	if (len == 0) {
	    continue;
	}

	/* Log the command we got */
	syslog (LOG_INFO, "Remote command: \"%s\"", cmd);

	/* The first part of the received line is expected to be a command. */
	if (strncmp (cmd, CMD_WRITE, strlen (CMD_WRITE)) == 0) {

	    /* Handle the WRITE command */
	    dowrite (getarg (cmd, strlen (CMD_WRITE)));

	} else if (strncmp (cmd, CMD_READ, strlen (CMD_READ)) == 0) {

	    /* Handle the READ command */
	    doread (getarg (cmd, strlen (CMD_READ)));

	} else if (strncmp (cmd, CMD_BYE, strlen (CMD_BYE)) == 0) {

	    /* Done */
	    break;

	} else {

	    /* Unknown command */
	    syslog (LOG_WARNING, "Unknown command: \"%s\"", cmd);
	    answer ("ERROR: Unknown command");

	}

    }

    /* Close the config file */
    CfgDone ();

    /* Done */
    syslog (LOG_NOTICE,
       	    "Connection terminated. read: %lu write: %lu",
	    rbytes,
	    wbytes);
    return 0;
}



