/*****************************************************************************/
/*                                                                           */
/*				   client.c				     */
/*                                                                           */
/*		     Code handling client specific things		     */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* (C) 1998     Ullrich von Bassewitz                                        */
/*              Wacholderweg 14                                              */
/*              D-70597 Stuttgart                                            */
/* EMail:       uz@musoftware.de                                             */
/*                                                                           */
/*                                                                           */
/* This software is provided 'as-is', without any express or implied         */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.                                    */
/*                                                                           */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:                            */
/*                                                                           */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.                                       */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.                      */
/* 3. This notice may not be removed or altered from any source              */
/*    distribution.                                                          */
/*                                                                           */
/*****************************************************************************/



#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <syslog.h>
#include <fnmatch.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "const.h"
#include "error.h"
#include "config.h"
#include "client.h"



/*****************************************************************************/
/*     	       	   	  	     Data  				     */
/*****************************************************************************/



/* IP address of the client in host byte order */
unsigned long clientaddr = 0x7F000001;	/* 127.0.0.1 */

/* Name or IP of the client in readable form */
char clientname [1024] = "";



/*****************************************************************************/
/*     	       	   	  	     Code  				     */
/*****************************************************************************/



static void iptoname (void)
/* Use the client ip address as client name */
{
    /* Use the address as name */
    sprintf (clientname,
	     "%u.%u.%u.%u",
	     (unsigned) (clientaddr >> 24) & 0xFF,
	     (unsigned) (clientaddr >> 16) & 0xFF,
	     (unsigned) (clientaddr >> 8) & 0xFF,
	     (unsigned) (clientaddr >> 0) & 0xFF);
}



void getclient (void)
/* Determine the name of the client. Abort on errors. */
{
    struct hostent* h;
    struct sockaddr_in client;
    long nodns;
    int len = sizeof (client);
    char* c;

    /* Determine the IP address of the client */
    if (getpeername (fileno (stdin), (struct sockaddr*) &client, &len) == -1) {
       	/* Some sort of error. To allow testing without having an IP
    	 * connection, check if stdin is not a socket, if so assume a
    	 * console connection on stdin and do not abort (but log an error).
    	 */
       	error ("Cannot determine client IP address (%d)", errno);
    	if (errno != ENOTSOCK) {
    	    exit (EXIT_FAILURE);
    	} else {
    	    /* Stdin is not a socket */
    	    strcpy (clientname, "stdin");
    	    return;
    	}
    }

    /* Remember the IP of the client */
    clientaddr = ntohl (client.sin_addr.s_addr);

    /* Check if doing DNS stuff is disallowed */
    CfgGetInt (0, "nodns", 0, &nodns);

    /* If we're allowed to do DNS resolution, map the IP into a hostname.
     * Otherwise use the string representation of the IP address as name.
     */
    if (nodns) {

        /* Use the address as name */
	iptoname ();

    } else {

	/* Try to map the address into a name */
	h = gethostbyaddr ((const char*)&client.sin_addr.s_addr,
     	    	    	   sizeof (client.sin_addr.s_addr),
     	    	    	   AF_INET);
	if (h == 0) {

	    /* Log the error */
	    if (h_errno == -1) {
		warning ("Cannot resolve client address: %m(%d)", errno);
	    } else {
		char* msg;
		switch (h_errno) {
		    case HOST_NOT_FOUND:
			msg = "host not found";
			break;

		    case NO_ADDRESS:
			msg = "no IP address for host";
			break;

		    case NO_RECOVERY:
			msg = "unrecoverable DNS error";
			break;

		    case TRY_AGAIN:
			msg = "temporary DNS error (try again)";
			break;

		    default:
			msg = "unknown return code";
			break;
		}

		warning ("Cannot resolve client address: %s(%d)", msg, h_errno);
	    }

	    /* Use the address as name */
	    iptoname ();

	} else {

	    /* We have a name */
	    strncpy (clientname, h->h_name, sizeof (clientname)-1);
	    clientname [sizeof (clientname)-1] = '\0';

	}
    }


    /* Security check: Don't accept whitespace here (should
     * never ever happen).
     */
    c = clientname;
    while (*c) {
       	if (isspace (*c++)) {
     	    syslog (LOG_CRIT, "Whitespace in client name!");
     	    exit (EXIT_FAILURE);
     	}
    }
}



static int inlist (char* buf)
/* Return YES if the client is in the host list given as argument. The
 * function uses strtok and will write into the argument buffer.
 */
{
    char* tok;
    unsigned q1, q2, q3, q4, bits;
    unsigned long ip, mask;


    while ((tok = strtok (buf, " \t")) != 0) {

	/* Use a NULL argument next time */
	buf = 0;

	/* If the first character is a digit, assume a standard dotted quad
	 * notation with an optional netmask.
	 */
	if (isascii (tok [0]) && isdigit (tok [0])) {

	    /* Set the defaults */
	    q1 = q2 = q3 = q4 = 0;
	    bits = 32;

	    /* Scan the value */
       	    if (sscanf (tok, "%u.%u.%u.%u/%u", &q1, &q2, &q3, &q4, &bits) > 4) {

		/* Get the IP */
		ip = (((unsigned long) q1) << 24) |
		     (((unsigned long) q2) << 16) |
		     (((unsigned long) q3) <<  8) |
		     (((unsigned long) q4) <<  0);

		/* Get the netmask */
		mask = (((unsigned long) 0xFFFFFFFF) << (32 - bits)) & 0xFFFFFFFF;

		/* Compare the given ip and the client ip */
		if ((ip & mask) == (clientaddr & mask)) {
		    /* We have a match */
		    return YES;
		}

	    }

	} else {

	    /* First character is not a digit, assume it's a client name
	     * (maybe with some wildcards). Try a text match.
	     */
	    if (fnmatch (tok, clientname, 0) == 0) {
		/* We have a match */
		return YES;
	    }

	}

    }

    /* No match */
    return FAILURE;
}



int clientaccess (const char* res, const char* allow, const char* deny)
/* Determine if the client is allowed access the service according to the
 * allow and deny lists in the section res.
 */
{
    char buf [1024];

    /* Read the "allow" entry in the config file. If such an entry exists,
     * only hosts listed there have access to the resource.
     */
    if (CfgGetStr (res, allow, "", buf, sizeof (buf)) != FAILURE) {
	/* The entry exists */
	return inlist (buf);
    }

    /* Read the "deny" entry in the config file. If such an entry exists,
     * client access is denied if the host is listed.
     */
    if (CfgGetStr (res, deny, "", buf, sizeof (buf)) != FAILURE) {
	/* The entry exists */
	return !inlist (buf);
    }

    /* There is neither an allow, nor a deny entry. Client access is allowed */
    return YES;
}



void answer (const char* format, ...)
/* Send an answer to the client, check for errors */
{
    char buf [1024];
    va_list ap;

    /* Create the output line */
    va_start (ap, format);
    vsnprintf (buf, sizeof (buf), format, ap);
    va_end (ap);

    /* Write the output line */
    if (puts (buf) == EOF) {
       	/* Error writing stdout - connection broken */
       	connbroken ();
    }

    /* Flush the output */
    fflush (stdout);
}



void okanswer (void)
/* Send an OK to the client */
{
    answer ("OK");
}



void erranswer (const char* msg, ...)
/* Send the given error message to the client */
{
    char buf [1024];
    va_list ap;

    /* Create the output line */
    va_start (ap, msg);
    vsnprintf (buf, sizeof (buf), msg, ap);
    va_end (ap);

    /* Write the output line */
    if (printf ("ERROR: %s\n", buf) == 0) {
       	/* Error writing stdout - connection broken */
       	connbroken ();
    }

    /* Flush the output */
    fflush (stdout);
}



