/*****************************************************************************/
/*									     */
/*				   backupc.c				     */
/*									     */
/*		  Backup client for use with OS/2, NT, Linux		     */
/*									     */
/*									     */
/*									     */
/* (C) 1998	Ullrich von Bassewitz					     */
/*		Wacholderweg 14						     */
/*		D-70597 Stuttgart					     */
/* EMail:	uz@musoftware.de					     */
/*									     */
/*									     */
/* This software is provided 'as-is', without any express or implied	     */
/* warranty.  In no event will the authors be held liable for any damages    */
/* arising from the use of this software.				     */
/*									     */
/* Permission is granted to anyone to use this software for any purpose,     */
/* including commercial applications, and to alter it and redistribute it    */
/* freely, subject to the following restrictions:			     */
/*									     */
/* 1. The origin of this software must not be misrepresented; you must not   */
/*    claim that you wrote the original software. If you use this software   */
/*    in a product, an acknowledgment in the product documentation would be  */
/*    appreciated but is not required.					     */
/* 2. Altered source versions must be plainly marked as such, and must not   */
/*    be misrepresented as being the original software.			     */
/* 3. This notice may not be removed or altered from any source		     */
/*    distribution.							     */
/*									     */
/*****************************************************************************/



#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>

#if defined(OS2)
/* Sorry for this mess - blame IBM */
#  include <io.h>
#  define BSD_SELECT
#  include <nerrno.h>
#  undef ENAMETOOLONG
#  undef ENOTEMPTY
#  include <netdb.h>
#  include <utils.h>
#  include <types.h>
#  include <sys/socket.h>
#  include <netinet/in.h>
#  include <sys/select.h>
#  undef BSD_SELECT
#  undef select
#  if defined(__WATCOMC__) && (__WATCOMC__ < 1100)
#    pragma aux (__system)  h_errno;	/* _System is missing in the IBM header file */
#  endif
#  define SOCK_CLOSE(fd)			soclose (fd)
#  define SOCK_SELECT(max,in,out,except,to)	bsdselect (max,in,out,except,to)
#  define SOCK_ERRCODE()			sock_errno()
#  define BIN_CRLF				"\r\n"
#elif defined(NT)
#  include <errno.h>
#  include <io.h>
#  include <winsock.h>
#  define SOCK_CLOSE(fd)			closesocket (fd)
#  define SOCK_SELECT(max,in,out,except,to)	select (max,in,out,except,to)
#  define SOCK_ERRCODE()			WSAGetLastError()
#  define BIN_CRLF				"\r\n"
#else
#  include <errno.h>
#  include <netdb.h>
#  include <unistd.h>
#  include <signal.h>
#  include <sys/time.h>
#  include <sys/socket.h>
#  include <netinet/in.h>
#  include <netinet/ip.h>
#  include <netinet/tcp.h>
#  ifdef AIX
#    include <sys/select.h>
#    include <strings.h>
#  endif
#  define SOCK_CLOSE(fd)			close (fd)
#  define SOCK_SELECT(max,in,out,except,to)	select (max,in,out,except,to)
#  define SOCK_ERRCODE()			errno
#  define BIN_CRLF				"\n"
#endif

#include "const.h"
#include "proto.h"



/*****************************************************************************/
/*				     data				     */
/*****************************************************************************/



/* Name of the program (set to argv [0]) */
static const char* progname = "backupc";

/* Host to connect to */
static const char* servername = "localhost";

/* Port to connect to */
static unsigned serverport = DEFAULTPORT;

/* Block size when writing */
static unsigned blocksize = 4096;

/* Socket handle */
static int sock = -1;

/* Output buffer data */
static unsigned		o_cur = 0;
static unsigned		o_max = 0;
static unsigned char*	o_buf = 0;

/* Input buffer data */
static unsigned		i_cur = 0;
static unsigned		i_max = 0;
static unsigned		i_lev = 0;
static unsigned char*	i_buf = 0;



/*****************************************************************************/
/*				    Basics				     */
/*****************************************************************************/



static void error (const char* format, ...)
/* Print an error message and abort */
{
    va_list ap;
    va_start (ap, format);
    vfprintf (stderr, format, ap);
    fprintf (stderr, "\n");
    va_end (ap);
    exit (EXIT_FAILURE);
}



static void* xmalloc (size_t size)
/* Allocate memory, abort if not possible */
{
    void* p = malloc (size);
    if (p == 0) {
	error ("Out of memory");
    }
    return p;
}



/*****************************************************************************/
/*			     OS/2 specific stuff			     */
/*****************************************************************************/



#ifdef OS2

static void init ()
{
    /* Initialize the socket package */
    sock_init ();

#ifdef __WATCOMC__
    /* Switch to binary mode for stdin and stdout */
    setmode (fileno (stdin), O_BINARY);
    setmode (fileno (stdout), O_BINARY);
#endif
}



static const char* sock_errmsg (long code)
/* Map a socket error code to a message */
{
    /* Under OS/2 we have no chance to get a error description, so we
     * have to hardcode the strings. Yuck!
     */
    switch (code) {
	case SOCEPERM:		return "Not owner";
	case SOCESRCH:		return "No such process";
	case SOCEINTR:		return "Interrupted system call";
	case SOCENXIO:		return "No such device or address";
	case SOCEBADF:		return "Bad file number";
	case SOCEACCES:		return "Permission denied";
	case SOCEFAULT:		return "Bad address";
	case SOCEINVAL:		return "Invalid argument";
	case SOCEMFILE:		return "Too many open files";
	case SOCEPIPE:		return "Broken pipe";
	case SOCEOS2ERR:	return "Os/2 error";
	case SOCEWOULDBLOCK:	return "Operation would block";
	case SOCEINPROGRESS:	return "Operation now in progress";
	case SOCEALREADY:	return "Operation already in progress";
	case SOCENOTSOCK:	return "Socket operation on non-socket";
	case SOCEDESTADDRREQ:	return "Destination address required";
	case SOCEMSGSIZE:	return "Message too long";
	case SOCEPROTOTYPE:	return "Protocol wrong type for socket";
	case SOCENOPROTOOPT:	return "Protocol not available";
	case SOCEPROTONOSUPPORT:return "Protocol not supported";
	case SOCESOCKTNOSUPPORT:return "Socket type not supported";
	case SOCEOPNOTSUPP:	return "Operation not supported on socket";
	case SOCEPFNOSUPPORT:	return "Protocol family not supported";
	case SOCEAFNOSUPPORT:	return "Address family not supported by protocol family";
	case SOCEADDRINUSE:	return "Address already in use";
	case SOCEADDRNOTAVAIL:	return "Can't assign requested address";
	case SOCENETDOWN:	return "Network is down";
	case SOCENETUNREACH:	return "Network is unreachable";
	case SOCENETRESET:	return "Network dropped connection on reset";
	case SOCECONNABORTED:	return "Software caused connection abort";
	case SOCECONNRESET:	return "Connection reset by peer";
	case SOCENOBUFS:	return "No buffer space available";
	case SOCEISCONN:	return "Socket is already connected";
	case SOCENOTCONN:	return "Socket is not connected";
	case SOCESHUTDOWN:	return "Can't send after socket shutdown";
	case SOCETOOMANYREFS:	return "Too many references: can't splice";
	case SOCETIMEDOUT:	return "Connection timed out";
	case SOCECONNREFUSED:	return "Connection refused";
	case SOCELOOP:		return "Too many levels of symbolic links";
	case SOCENAMETOOLONG:	return "File name too long";
	case SOCEHOSTDOWN:	return "Host is down";
	case SOCEHOSTUNREACH:	return "No route to host";
	case SOCENOTEMPTY:	return "Directory not empty";
	default:		return "Unknown return code";
    }
}

#endif



/*****************************************************************************/
/*			      NT specific stuff				     */
/*****************************************************************************/



#ifdef NT

static void init ()
{
    /* Need to initialize the socket package */
    WSADATA Data;
    if (WSAStartup (MAKEWORD (2, 0), &Data) != 0) {
	error ("Cannot initialize Windows sockets");
    }

#ifdef __WATCOMC__
    /* Switch to binary mode for stdin and stdout */
    setmode (fileno (stdin), O_BINARY);
    setmode (fileno (stdout), O_BINARY);
#endif
}



static const char* sock_errmsg (long code)
/* Map a socket error code to a message */
{
    /* Under NT we have no chance to get a error description, so we
     * have to hardcode the strings. Yuck!
     */
    switch (code) {
	case WSAEINVAL:		 return "Invalid parameter";
	case WSASYSNOTREADY:	 return "System not ready";
	case WSAVERNOTSUPPORTED: return "Winsock version not supported";
	case WSANOTINITIALISED:  return "Winsock module not initialized";
	case WSAENETDOWN:	 return "Network is down";
	case WSAHOST_NOT_FOUND:  return "Host not found";
	case WSATRY_AGAIN:	 return "Temporary error - try again";
	case WSANO_RECOVERY:	 return "Fatal network error";
	case WSANO_DATA:	 return "Host has no address";
	case WSAENETRESET:	 return "Connection dropped - must be reset";
	case WSAENOBUFS:	 return "Buffer deadlock - windows error";
	case WSAECONNABORTED:	 return "Connection was aborted";
	case WSAECONNRESET:	 return "Connection was reset by remote side";
	case WSAEADDRINUSE:	 return "Address is already in use";
	case WSAEADDRNOTAVAIL:	 return "Address is unavailable";
	case WSAENETUNREACH:	 return "Network is unreachable";
	case WSAETIMEDOUT:	 return "Connection timed out";
	case WSAECONNREFUSED:	 return "Connection refused";
	case WSAEHOSTDOWN:	 return "Host is down";
	case WSAEHOSTUNREACH:	 return "Host is unreachable";
	default:		 return "Unknown return code";
    }
}

#endif



/*****************************************************************************/
/*			     Unix specific stuff			     */
/*****************************************************************************/



#if !defined(OS2) && !defined(NT)

static void ignoresig (int sig)
/* Ignore the given signal */
{
    struct sigaction action, oldaction;
    action.sa_handler = SIG_IGN;
    sigemptyset (&action.sa_mask);
    action.sa_flags   = SA_RESTART;	/* Restart interrupted system calls */
    if (sigaction (sig, &action, &oldaction) != 0) {
	error ("Error ignoring signal %d: %s", sig, strerror (errno));
    }
}



static void init ()
{
    /* We must ignore the SIGPIPE signal under Unix */
    ignoresig (SIGPIPE);
}



static const char* sock_errmsg (long code)
/* Map a socket error code to a message */
{
    /* Under real operating systems, this is easy ... */
    return strerror ((int) code);
}

#endif



/*****************************************************************************/
/*			 buffer for the outgoing data			     */
/*****************************************************************************/



/* Since the code must run under other operating systems than Linux (and
 * Linux clones), we have to do the buffering ourselves.
 */



static void oflush (void)
/* Flush the output buffer */
{

    unsigned char* b = o_buf;
    while (o_cur) {

	int rc;
	int size = o_cur;

#ifdef OS2
	/* OS/2 cannot write more than 0x7FFF bytes */
	if (size > 0x7FFF) {
	    size = 0x7FFF;
	}
#endif

	/* Write out one chunk */
	rc = send (sock, b, size, 0);

	/* Check for an error */
	if (rc == -1) {
	    long code = SOCK_ERRCODE ();
	    error ("Cannot write: %s(%ld)", sock_errmsg (code), code);
	}

	/* Next chunk */
	b     += rc;
	o_cur -= rc;

    }
}



static void oputc (int c)
/* Put one character into the output buffer */
{
    /* Flush the output buffer if needed */
    if (o_cur >= o_max) {
	oflush ();
    }

    /* Put the character into the buffer */
    o_buf [o_cur++] = c;
}



static void oputs (const char* str)
/* Put a string int the output buffer */
{
    while (*str) {
	oputc (*str);
	++str;
    }
}



static char* oputbuf (char* buf, unsigned count)
/* Write the contents of the buffer to the output */
{
    if (o_max - o_cur >= count) {
	/* There's room enough in the buffer */
	memcpy (o_buf + o_cur, buf, count);
	o_cur += count;
    } else {
	/* No room, do it char by char */
	char* b = buf;
	while (count--) {
	    oputc (*b++);
	}
    }
    return buf;
}



static void ofree (void)
/* Free the output buffer */
{
    if (o_buf) {
	free (o_buf);
	o_buf = 0;
	o_max = 0;
	o_cur = 0;
    }
}



static void oalloc (unsigned size)
/* Allocate the output buffer */
{
    /* Free an eventually existing buffer */
    ofree ();

    /* Allocate memory */
    o_buf = xmalloc (size);
    o_max = size;
    o_cur = 0;
}



static void iread (void)
/* Read new data into the (empty) input buffer */
{
    int rc;
    int size = i_max;

#ifdef OS2
    /* OS/2 cannot handle blocks larger than 0x7FFF bytes */
    if (size > 0x7FFF) {
	size = 0x7FFF;
    }
#endif

    /* Read a chunk into the buffer */
    rc = recv (sock, i_buf, size, 0);

    /* Check for errors */
    if (rc == -1) {
	long code = SOCK_ERRCODE ();
	error ("Cannot read: %s(%ld)", sock_errmsg (code), code);
    }

    /* Remember what we have */
    i_lev = rc;
    i_cur = 0;
}



static int igetc (void)
/* Get a character from the input buffer */
{
    /* Read new data if needed */
    while (i_cur >= i_lev) {
	iread ();
    }

    /* Get the next char and return it */
    return i_buf [i_cur++];
}



static char* igets (char* buf, unsigned size)
/* Get a line from the input */
{
    int c;
    int i = 0;

    /* Read the string */
    while ((c = igetc ()) != '\n') {
	if (i < size-1) {
	    buf [i++] = c;
	}
    }

    /* And terminate it */
    buf [i] = '\0';

    /* Done */
    return buf;
}



static char* igetbuf (char* buf, unsigned count)
/* Read count characters from input into buf */
{
    if (i_lev - i_cur >= count) {
	/* There are enough characters in the buffer, access it directly */
	memcpy (buf, i_buf + i_cur, count);
	i_cur += count;
    } else {
	/* Get it char by char */
	char* b = buf;
	while (count--) {
	    *b++ = igetc ();
	}
    }
    return buf;
}



static void ifree (void)
/* Free the input buffer */
{
    if (i_buf) {
	free (i_buf);
	i_buf = 0;
	i_max = 0;
	i_lev = 0;
	i_cur = 0;
    }
}



static void ialloc (unsigned size)
/* Allocate the input buffer */
{
    /* Free an eventually existing buffer */
    ifree ();

    /* Allocate memory */
    i_buf = xmalloc (size);
    i_max = size;
    i_lev = 0;
    i_cur = 0;
}



/*****************************************************************************/
/*				     code				     */
/*****************************************************************************/



static void usage (void)
/* Print usage information and exit */
{
    error ("Usage: %s [options]\n"
	   "Option list (order *is* significant):\n"
	   "\t-b n\tUse n KB as block size\n"
	   "\t-h name\tConnect to host name (default is localhost)\n"
	   "\t-l\tList available resources on the server\n"
	   "\t-p port\tConnect to port instead of %u\n"
	   "\t-r dev\tRead the given device\n"
	   "\t-V\tPrint version number\n"
	   "\t-w dev\tWrite to the given device\n",
	   progname,
	   serverport);
}



static const char* getarg (const char* cmd, unsigned len)
/* Given a command line with command length len, return a pointer to the
 * command argument(s).
 */
{
    /* Skip the command */
    cmd += len;

    /* Skip whitespace */
    while (isascii (*cmd) && isspace (*cmd)) {
	++cmd;
    }

    /* Return the pointer */
    return cmd;
}



static void getanswer (char* buf, unsigned size, const char* expected)
/* Read an answer and handle ERROR returns. If expected is not NULL, check
 * if the answer matches expected and abort if not.
 */
{
    static const char RESP_ERROR [] = "ERROR:";

    /* Read a string from remote */
    igets (buf, size);

    /* Check for errors */
    if (strncmp (buf, RESP_ERROR, strlen (RESP_ERROR)) == 0) {
	/* Some problem on the remote side */
	error ("Server error: %s", getarg (buf, strlen (RESP_ERROR)));
    }

    /* Check if the answer matches the expected answer */
    if (expected) {
	if (strncmp (buf, expected, strlen (expected)) != 0) {
	    error ("Unexpected answer from server: %s", buf);
	}
    }
}



static int srvconnect (void)
/* Connect to the server or fail */
{
    struct hostent* h;
    struct sockaddr_in srv;
    int s;

    /* Get the host entry for the specified host */
    h = gethostbyname (servername);
    if (h == 0) {
	char* msg;
	switch (h_errno) {
	    case HOST_NOT_FOUND:
		msg = "host not found";
		break;

	    case NO_ADDRESS:
		msg = "no IP address for host";
		break;

	    case NO_RECOVERY:
		msg = "unrecoverable DNS error";
		break;

	    case TRY_AGAIN:
		msg = "temporary DNS error (try again)";
		break;

	    default:
		msg = "unknown return code";
		break;
	}
	error ("Cannot resolve server name: %s", msg);
    }

    /* Setup the sin structure */
    memset (&srv, 0, sizeof (srv));
    srv.sin_family	 = AF_INET;
    srv.sin_addr.s_addr  = *((u_long*) h->h_addr);
    srv.sin_port	 = htons (serverport);

    /* Create the socket */
    s = socket (AF_INET, SOCK_STREAM, 0);
    if (s < 0) {
	long code = SOCK_ERRCODE ();
	error ("Cannot create socket: %s(%ld)", sock_errmsg (code), code);
    }

    /* Connect to the server */
    if (connect (s, (struct sockaddr*) &srv, sizeof (srv)) < 0) {
	long code = SOCK_ERRCODE ();
	error ("Cannot connect to server: %s(%ld)", sock_errmsg (code), code);
    }

    /* Return the socket handle */
    return s;
}



static void dobye (void)
/* Close the connection gracefully */
{
    /* Tell the server that we're about to shutdown */
    oputs (CMD_BYE);
    oputc ('\n');
    oflush ();

    /* Close the socket */
    SOCK_CLOSE (sock);

    /* Free any allocated buffers */
    ifree ();
    ofree ();
}



static void okbye (void)
/* Check last response and close the connection gracefully */
{
    char buf [256];

    /* Flush current output */
    oflush ();

    /* Wait for the last answer */
    getanswer (buf, sizeof (buf), "OK");

    /* Shutdown */
    dobye ();
}



static void dowrite (const char* dev)
/* Handle the WRITE command */
{
    char buf [256];


    /* Open the connection to the server */
    sock = srvconnect ();

    /* Create the input and output buffers */
    ialloc (256);
    oalloc (blocksize);

    /* Send the command */
    oputs (CMD_WRITE);
    oputc (' ');
    oputs (dev);
    oputc ('\n');
    oflush ();

    /* Wait for the OK */
    getanswer (buf, sizeof (buf), "OK");

    /* Send data */
    while (1) {

	/* Get a chunk from stdin */
	int count = fread (buf, 1, 255, stdin);
	if (count == 0) {
	    break;
	}

	/* Write the block to the output buffer */
	oputc (count);
	oputbuf (buf, count);
    }

    /* Add the end marker */
    oputc (0);

    /* Close the connection */
    okbye ();
}



static void doread (const char* dev)
/* Handle the READ command */
{
    int count;
    char buf [256];


    /* Open the connection to the server */
    sock = srvconnect ();

    /* Create the input and output buffers */
    ialloc (blocksize);
    oalloc (256);

    /* Send the command */
    oputs (CMD_READ);
    oputc (' ');
    oputs (dev);
    oputc ('\n');
    oflush ();

    /* Wait for the OK */
    getanswer (buf, sizeof (buf), "OK");

    /* Read data */
    while ((count = igetc ()) != 0) {

	igetbuf (buf, count);
	if (fwrite (buf, 1, count, stdout) != count) {
	    error ("Cannot write to stdout");
	}

    }

    /* Close the connection */
    okbye ();
}



static void dolist (void)
/* List available server resources */
{
    char buf [256];

    /* Open the connection to the server */
    sock = srvconnect ();

    /* Create the input and output buffers */
    ialloc (1024);
    oalloc (256);

    /* Send the command */
    oputs (CMD_LIST);
    oputc ('\n');
    oflush ();

    /* Read and output anything until we get an ok */
    while (1) {
	/* Get next string */
	getanswer (buf, sizeof (buf), 0);

	/* Check for "OK" */
	if (strcmp (buf, "OK") == 0) {
	    break;
	}

	/* Line must start with a dash */
	if (buf [0] != '-') {
	    error ("Unexpected server response: \"%s\"", buf);
	}

	/* Print the line without the dash.
	 * Beware: stdout is in binary mode.
	 */
	printf ("%s%s", &buf [1], BIN_CRLF);
    }

    /* Close the connection */
    dobye ();
}



static char* getoptarg (int* index, char* argv [])
/* Get an argument for the one letter option in argument number *index */
{
    char* arg = &(argv [*index][2]);
    if (*arg == '\0') {
	/* Use next argument */
	arg = argv [*index+1];
	if (arg == 0) {
	    /* End of argument s*/
	    error ("Missing argument for option %s", argv [*index]);
	}
	*index += 1;
    }
    return arg;
}



int main (int argc, char* argv [])
{
    int i;


    /* Initialize variables */
    progname = argv [0];

    /* Initialize the socket package if needed */
    init ();

    /* Parse program options */
    i = 1;
    while (i < argc) {

	/* Get next argument */
	char* arg = argv [i];

	/* Must be an option */
	if (arg [0] != '-') {
	    usage ();
	}

	switch (arg [1]) {

	    case 'b':
		blocksize = atoi (getoptarg (&i, argv)) * 1024;
		break;

	    case 'h':
		servername = getoptarg (&i, argv);
		break;

	    case 'l':
		dolist ();
		break;

	    case 'p':
		serverport = atoi (getoptarg (&i, argv));
		break;

	    case 'r':
		doread (getoptarg (&i, argv));
		break;

	    case 'V':
		fprintf (stderr, "Version: %s\n", VERSION);
		break;

	    case 'w':
		dowrite (getoptarg (&i, argv));
		break;

	    default:
		usage ();
		break;

	}

	/* Next arg */
	++i;
    }


    /* Done */
    return 0;
}




