/*****************************************************************************
 * $Id: ak_dir.c,v 2.19 1996/09/03 20:30:35 ak Exp $
 *****************************************************************************
 * $Log: ak_dir.c,v $
 * Revision 2.19  1996/09/03 20:30:35  ak
 * Allow cluster size variation.
 *
 * Revision 2.18  1996/07/05 21:04:47  ak
 * Decoder fixes.
 *
 * Revision 2.17  1996/06/06 15:33:27  ak
 * ZLIB based compression engine.
 *
 * Revision 2.16  1995/03/06 23:48:21  ak
 * Bugfix for more than 10 volumes.
 *
 * Revision 2.15  1995/02/27  20:18:56  ak
 * Missing parentheses caused arithmetic overflow and negative record
 * offsets in the tape directory.
 *
 * Revision 2.14  1995/01/01  00:13:26  ak
 * Umlaute im QFA-File nicht maskieren.
 * Reblock bei compressed.
 *
 * Revision 2.13  1994/11/08  20:12:13  ak
 * Optional case-sensitive compare.
 * Moved include of port.h to tar.h.
 *
 * Revision 2.12  1994/10/19  01:56:02  ak
 * Tape directory support in buffered mode.
 * Volume labels in tape directory were wrong.
 * Alias -Q for --buffered.
 * Priority control for buffered mode.
 *
 * Revision 2.11  1994/09/04 21:59:22  ak
 * QFA failed if a huge file (>= 1e8 bytes) caused a line shift.
 *
 * Revision 2.10  1994/08/09 16:20:42  ak
 * Bugfix QFA mit Tape-Dir von 2.12.
 *
 * Revision 2.9  1994/07/05 18:44:42  edvkai
 * 2.35: Posix support for long filenames and magic name.
 *
 * Lots of changes in filename handling, I hope I found all places. Added
 * decode_filename() with result in var 'filename' for this. header.name
 * should no longer be used. The diffarch kludge FILENAME is no longer
 * necessary.
 *
 * Treatment of links is not up-to-date, I fear - what's defined by Posix
 * about long link names?
 *
 * Due to NONAMES, prior versions created "oldarch" style archives without
 * magic name. Now magic is "ustar\0""00" for Posix and "ustar  \0" else.
 *
 * Unused header space is used to support multi-volume archives in Posix -
 * THIS IS SPECIFIC TO GTAK.
 *
 * Note that Posix conflicts with atime/ctime.
 *
 * Revision 2.8  1994/07/04 02:03:07  ak
 * Rewrote QFA directory entry parser to allow support all versions of
 * tape directories since GTAK 1.0.
 *
 * Further reduced amount of redundant text in tape directory.
 *
 * Version 2.34.
 *
 * Revision 2.7  1994/06/18 21:33:39  ak
 * Removed some upper/lowercase dependancies on filenames. Volume
 * names still are case-sensitive.
 *
 * QFA failed when the buffer was not empty and the requested record was
 * located in the very next block.
 *
 * Revision 2.6  1993/12/14 22:12:15  ak
 * Bugfix QFA. Physical seek didn't flush buffer.
 *
 * Revision 2.5  1993/12/03  15:01:22  edvkai
 * QFA bugfix.
 *
 * Revision 2.4  1993/11/30  23:16:38  ak
 * Accept old tape directories.
 *
 * Revision 2.3  1993/11/29  17:02:30  edvkai
 * Multivolume QFA fixes.
 * Smaller size of tape directory.
 *
 * Revision 2.2  1993/11/25  20:23:20  edvkai
 * Major QFA changes.
 *
 * QFA now supports both block id and relative addressing. Relative
 * addressing is more efficient with short distances. 4mm and 8mm devices
 * do well with relative addressing only. For the sake of QIC tapes, TAR
 * optimizes QFA using block ids for large distances.
 *
 * The format of the tape-directory changed. No longer compatible to prior
 * versions. Maybe I'll add a utility for conversion.
 *
 * Rmt interface changed. Now using block numbers instead of byte offsets
 * for seek. Previously the lseek limitation to 2GB limited QFA archive
 * capacity. Changed rmtlseek to rmtseek to indicate that it is different.
 *
 * QFA should be able to handle multi-volume archives now.
 *
 * If a volume name is specified on the command line, QFA skips archives
 * having a different volume name.
 *
 * Code for header pretty-printing cleaned. Doesn't depend on hstat any
 * longer and requires less globals. No explanation text behind names in
 * tape directory.
 *
 * A few bugfixes.
 *
 * Revision 2.1  1993/08/08  19:06:00  ak
 * Merge of network TAR with 2.12.
 *
 * Revision 1.1.1.2  1993/08/08  18:43:04  ak
 * - Network tape access.
 * - Dynamically loaded modules for disk/scsi/network interfaces.
 *
 * Revision 1.3  1992/10/29  10:26:35  ak
 * *** empty log message ***
 *
 * Revision 1.2  1992/09/12  15:57:09  ak
 * - Usenet patches for GNU TAR 1.10
 * - Bugfixes and patches of Kai Uwe Rommel:
 *         filename conversion for FAT
 *         EMX 0.8e
 *         -0..1 alias for a: b:
 *         -2..7 alias for +++TAPE$x
 *
 * Revision 1.1  1992/09/02  20:07:32  ak
 * Initial revision
 *
 *****************************************************************************/

static char *rcsid = "$Id: ak_dir.c,v 2.19 1996/09/03 20:30:35 ak Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "tar.h"
#include "rmt.h"

#define CLUSTER		

extern union record *	head;		/* Points to current tape header */
extern struct stat 	hstat;		/* Stat struct corresponding */
extern int		head_standard;	/* Tape header is in ANSI format */
extern long		baserec;
extern int		volno;

int		f_cluster  = 1;		/* default cluster size (MB) */

static long	lastseek   = -99999;	/* last block id */
static int	lastvol    = -1;	/* last volume handled by seek */
static int	resync	   = 1;		/* current position unknown */
static int	error_flag = 0;

int		map_read;	/* signals map is read, not written */

static char *	mapname;	/* temp file name for mapfile */
static FILE *	mapfile;	/* the map file written to while archiving */
static FILE *	mapfile2;	/* the final map file (in split qfa mode) */
static long *	qfamap;		/* record/id map */
static int	qfasize;	/* allocated size of .. */
static int	qfalen;		/* actual size of .. */

typedef struct {
	char	name[60];
	int	baselen;
	int	volno;
	int	delta;
	long	blk, rec;
} Volume;
static Volume *	volmap	= 0;			/* volume map */
static Volume *	volmax	= 0;
static int	volx	= 0;
static int	volcnt	= 0;

/*
 * Strip trailing garbage in old tape directories.
 */
static void
strip_old(char *line, char off)
{
	char *p, *q;

	switch (line[off]) {
	case 'V': case 'M': case 'N':
		q = line + strlen(line);
		while (isspace(*(q-1)))
			--q;
		if (*(q-1) == '-' && *(q-2) == '-') {
			q -= 2;
			while (q > line+2 && !(*(q-1) == '-' && *(q-2) == '-'))
				--q;
			if (q > line+2) {
				*(q-2) = '\n';
				*(q-1) = '\0';
			}
		}
		break;
	case 'A':
		q = line + strlen(line);
		while (isspace(*(q-1)))
			--q;
		if (*(q-1) == ')') {
			q -= 1;
			while (q > line+2 && !(*(q-1) == '(' && *(q-2) == ' '))
				--q;
			if (q > line+2) {
				*(q-2) = '\n';
				*(q-1) = '\0';
			}
		}
		break;
	}
}

static char *
decode_100(char *line, long *blk, long *off, long *rec, char *tag)
{
	char *p;
	long n;
	static long vol;

	n = strtoul(line+4, NULL, 10);

	if (memcmp(line, "vol ", 4) == 0) {
		vol = n+1;
		return NULL;
	}

	switch (*tag = line[16]) {
	case 'V': case 'M': case 'N': case 'A':
		strip_old(line, 16);
	}

	if (line[0] == 'V') {
		*blk = vol;
		*off = 0;
		*rec = 0;
	} else {
		*blk = vol + n-1;
		*off = 0;
		*rec = n-1;
	}
	return line + 58;
}

static char *
decode_200(char *line, long *blk, long *off, long *rec, char *tag)
{
	char *p;

	switch (*tag = line[19]) {
	case 'V': case 'M': case 'N': case 'A':
		strip_old(line, 19);
	}

	*blk = strtol(line+4, &p, 10);
	*off = (*p == '+') ? strtol(p+1, &p, 10) : 0;
	*rec = -1;
	return line + 61;
}

static char *
decode_222(char *line, long *blk, long *off, long *rec, char *tag)
{
	char *p;

	if (memcmp(line, "blk=", 4) == 0) {
		*blk = strtol(line+4, &p, 10);
		*off = (*p == '+') ? strtol(p+1, &p, 10) : 0;
		if (*p == ',')
			++p;
	} else {
		*blk = -1;
		*off = 0;
	}
	if (memcmp(p, "rec=", 4) == 0) {
		*rec = strtol(p+4, &p, 10);
		if (*p++ == '\t') {
			*tag = *p;
			return p + 29;
		}
	}
	msg("Invalid tape directory, quitting.");
	exit(EX_BADARCH);
}

static char *
decode_234(char *line, long *blk, long *off, long *rec, char *tag)
{
	char *p;

	for (p = line; *p != ',' && *p != '\t' && *p != '\0'; ++p)
		;
	if (*p == ',') {
		*blk = strtol(line, &p, 10);
		*off = (*p == '+') ? strtol(p+1, &p, 10) : 0;
		if (*p == ',')
			++p;
	} else {
		*blk = -1;
		*off = 0;
		p = line;
	}
	if (isdigit(*p)) {
		*rec = strtol(p, &p, 10);
		if (*p++ == '\t') {
			*tag = *p;
			/* beware of line-shift due to huge file (>= 1e8) */
			do ++p; while (isspace(*p));
			while (*p && !isspace(*p)) ++p;
			return p + 19;
		}
	}
	msg("Invalid tape directory, quitting.");
	exit(EX_BADARCH);
}

/*
 * Decode tape directory entry.
 * Return filename, NULL if null entry.
 */
static char *
decode(char *line, long *blk, long *off, long *rec, char *tag)
{
	char *name, *p;

	if (isdigit(line[0]) || line[0] == '-')
		name = decode_234(line, blk, off, rec, tag);
	else if (memcmp(line, "blk=", 4) == 0
	      || memcmp(line, "rec=", 4) == 0)
		name = decode_222(line, blk, off, rec, tag);
	else if (memcmp(line, "blk ", 4) == 0)
		name = decode_200(line, blk, off, rec, tag);
	else if (memcmp(line, "vol ", 4) == 0
	      || memcmp(line, "rec ", 4) == 0
	      || line[0] == 'V')
		name = decode_100(line, blk, off, rec, tag);
	else {
		msg("Invalid tape directory, quitting.");
		exit(EX_BADARCH);
	}
	if (name) {
		for (p = name + strlen(name); p > name && isspace(*(p-1)); --p)
			;
		*p = '\0';
	}
	return name;
}
 
/*
 * Read volume map.
 */
void
read_volume_map()
{
	char line[60 + FILENAME_MAX];

	volx = 0;
	rewind(mapfile);
	while (fgets(line, 60 + FILENAME_MAX, mapfile)) {
		long blk = -1, off, rec;
		char tag, *name, *p;
		Volume *volp;

		if ((name = decode(line, &blk, &off, &rec, &tag)) == NULL)
			continue;

		if (tag == 'M')
			++volp->delta;
		if (tag != 'V')
			continue;

		if (!volmap) {
			volcnt = 50;
			volmap = malloc(volcnt * sizeof(Volume));
		} else if (volx == volcnt) {
			volcnt *= 2;
			volmap = realloc(volmap, volcnt * sizeof(Volume));
		}
		if (!volmap) {
			msg("Out of memory, quitting");
			exit(EX_SYSTEM);
		}
		volp = &volmap[volx++];
		volp->blk = blk;
		volp->rec = rec;
		volp->delta = 1;

		if (p = strstr(name, " Volume ")) {
			volp->baselen = p - name;
			volp->volno = strtol(p+8, NULL, 10);
		} else {
			volp->baselen = strlen(name);
			volp->volno = 0;
		}
		memcpy(volp->name, name, volp->baselen + 1);
	}
	if (volx)
		volmax = &volmap[volx-1];
}

/*
 * Volume management.
 * mode = 0: relative
 * mode = 1: physical seek for file, no volume sync required
 * mode = 2: physical seek to volume
 */
int
change_volume(Volume *volp, int mode)
{
	if (f_multivol && volp->volno > 0) {
		static int firsttime = 1;
		if (firsttime || volno != volp->volno) {
			firsttime = 0;
			lastvol = volno = volp->volno - 1; /* incremented by new_volume() */
			new_volume(1);
			baserec = 0;
			lastseek = 0;
			ar_last = ar_record = ar_block;
			resync = 1;
		}
	}

	switch (mode) {

	case 2:	if (f_verbose)
			msg("Seek to physical block %ld", volp->blk);
		if (rmtseek(archive, volp->blk, 3) < 0) {
			msg_perror("Seek error on %s, physical block %ld", ar_file, volp->blk);
			if (++error_flag > 10) {
				msg("Too many errors, quitting.");
				exit(EX_BADARCH);
			}
		}
		lastseek = volp->blk;
		baserec = 0;
		ar_last = ar_record = ar_block;
		break;

	case 0:	if (f_multivol && volno != lastvol) {
			/* volume has been changed by reading a continued file */
			if (f_verbose)
				msg("Rewind");
			if (rmtseek(archive, 0L, 5) < 0) { /* rewind */
				msg_perror("Could not rewind tape, quitting");
				exit(EX_BADARCH);
			}
		}
		if (f_verbose)
			msg("Looking for volume \"%s\"", volp->name);
		for (;;) {
			baserec = 0;
			ar_last = ar_record = ar_block;
			if (read_header() == 1 && head->header.linkflag == 'V') {
				head->header.name[NAMSIZ-1] = '\0';
				if (f_verbose)
					msg("Found volume \"%s\"", head->header.name);
				if (strcmp(head->header.name, volp->name) == 0) {
					userec(head);
					return 1;
				}
			}
			if (rmtseek(archive, 0L, 2) < 0) {
				msg("Volume \"%s\" not found", volp->name);
				return 0;
			}
		}
	}
	return 1;
}

/*
 * Random archive access.
 */
int
seek_record(long blk, long off, long rec)
{
	long r, records, blocks, delta;
	char blkid[50];

	/*
	 * Seek to requested tape position. Make sure the buffer
	 * position (baserec) is always aligned to a block boundary.
	 */

	/* Compute distance from beginning of current block. */
	delta = rec - baserec;

	/* Check for absolute seeks - treshold is a distance of 10 clusters. */
	if (blk >= 0
	 && (f_map_mode == 'a' && (resync || delta > 10 * f_cluster * 1024 * 2)
	  || rec < 0 && blk != lastseek)) {
		sprintf(blkid, "physical block %ld+%ld", blk, off);
		if (f_verbose)
			msg("Seek to physical block %ld", blk);
		if (rmtseek(archive, blk, 3) < 0)
			goto seek_error;
		resync = 0;
		lastseek = blk;
		baserec = (rec < 0) ? 0 : rec - off;
		delta = off;
		ar_last = ar_record = ar_block;
	} else if (rec >= 0) {
		sprintf(blkid, "record %ld", rec);
	} else {
		sprintf(blkid, "physical block %ld+%ld", blk, off);
		delta = off - baserec;
	}

	/*
	 * Compute distance in blocks and records, still counted
	 * from beginning of current block, not from current record.
	 *
	 * If the requested record is located within the current
	 * or the very next block, just skip records. Max record skip
	 * is 2 * blocking - 1.
	 *
	 * If the requested record is located beyond the next block,
	 * discard the current block, correct baserec, seek blocks
	 * and skip the remaining records.
	 */
	records = delta % blocking;
	blocks  = delta / blocking;
	if (blocks > 0 && ar_last > ar_block) {
		assert(ar_last - ar_block == blocking);
		if (--blocks)
			baserec += blocking;	/* beyond next block */
		else
			records += blocking;	/* in very next block */
	}
	if (blocks > 0) {
		if (f_verbose)
			msg("Skip %ld tar blocks", blocks);
		if (rmtseek(archive, blocks, 1) < 0)
			goto seek_error;
		baserec += blocks * blocking;
		ar_last = ar_record = ar_block;	/* discard */
	} else {
		records -= ar_record - ar_block;
	}
	if (records > 0) {
		if (f_verbose)
			msg("Skip %ld tar records", records);
		while (records-- > 0)
			userec(findrec());
	}
	return 1;

seek_error:
	msg_perror("Seek error on %s, %s", ar_file, blkid);
	if (++error_flag > 10) {
		msg("Too many errors, quitting.");
		exit(EX_BADARCH);
	}
	return 0;
}

/*
 * Random file access.
 */
void
seek_exec(void (*do_something)(), char *name, long blk, long off, long rec)
{
	if (blk < 0 && rec < 0) {
		msg("File %s has neither block nor record id, quitting", name);
		exit(EX_ARGSBAD);
	}

	if (!seek_record(blk, off, rec))
		return;

	if (read_header() == 1) {
		decode_filename(head);
		(*do_something)();
	} else {
		msg("File %s has invalid header, wrong map file?", name);
		if (++error_flag > 10) {
			msg("Too many errors, quitting.");
			exit(EX_BADARCH);
		}
	}
	return;

}

/*
 * Main loop for random access.
 */
void
seek_and(do_something)
	void (*do_something)();
{
	struct name *nlp;
	char	line[60 + FILENAME_MAX], *linep;
	char	volume[120];
	long	volblk = -1, voloff, volrec;
	int	len, newvol, delta;
	Volume	*logvolp, *realvolp;

	read_volume_map();

	name_gather();		/* Gather all the names */
	open_archive(1);	/* Open for reading */

	logvolp = realvolp = volmap ? volmap-1 : 0;
	newvol = f_multivol;
	delta = 0;		/* When a new volume is opened, files
				   already placed in the buffer are up to
				   two blocks off in record number. */
	rewind(mapfile);
	while (fgets(line, 60 + FILENAME_MAX, mapfile)) {
		long blk = -1, off, rec;
		char tag, *name;

		if ((name = decode(line, &blk, &off, &rec, &tag)) == NULL)
			continue;

		if (tag == 'V') {
			assert(logvolp != NULL);
			if (logvolp == realvolp)
				++realvolp;
			logvolp = realvolp;
				
			assert(logvolp <= volmax);
			assert(strncmp(name, logvolp->name, strlen(logvolp->name)) == 0);
			newvol = 1;
			resync = 1;
			delta = 0;
			continue;
		}
		if (tag == 'M')
			continue;

		if (f_volhdr
		 && realvolp
		 && (strlen(f_volhdr) != realvolp->baselen
		  || memcmp(f_volhdr, realvolp->name, realvolp->baselen) != 0))
			/* wrong volume */
			continue;

		len = strlen(name);

		/* look if name is mentioned in command line */
		for (nlp = namelist; nlp; nlp = nlp->next) {
			/* regular expression? */
			if (nlp->regexp) {
				if (wildmat(name, nlp->name))
					goto match;
				continue;
			}

			/* plain name */
			if (nlp->length <= len
			 && (name[nlp->length] == '\0' || name[nlp->length] == '/')
			 && strnIcmp(name, nlp->name, nlp->length) == 0)
			 	goto match;
			continue;

		match:	/* do some look-ahead to find out if this file
			   didn't fit onto the tape and is placed on a
			   followup tape */
			if (f_multivol) {
				if (rec < 0)
					msg("Multivolume QFA is unreliable with old tape directory");
				else if (realvolp
				      && realvolp < volmax
				      && rec >= (realvolp+1)->rec
				      && realvolp->baselen == (realvolp+1)->baselen
				      && memcmp(realvolp->name, (realvolp+1)->name, realvolp->baselen) == 0) {
					newvol = 1;
					++realvolp;
					delta += realvolp->delta;
				}
			}
			if (newvol && realvolp) {
				change_volume(realvolp,	logvolp == realvolp
					? (f_map_mode == 'a' && blk >= 0 ? 1 : 0)
					: (f_map_mode == 'a' && realvolp->blk >= 0 ? 2 : 0));
				newvol = 0;
			}
			if (realvolp && realvolp != logvolp)
				/* physical block id incorrect */
				seek_exec(do_something, name, -1, -1,
					rec + delta - realvolp->rec);
			else if (realvolp && rec >= 0)
				seek_exec(do_something, name, blk, off,
					rec - realvolp->rec);
			else
				seek_exec(do_something, name, blk, off, rec);
			if (tag != 'A' && tag != 'L')
				nlp->found = 1;
		}
	}

	close_archive(1);
	names_notfound();		/* Print names not found */
}

/*****************************************************************************/

void
create_volume_tag()
{
#ifdef TAPE_IO
#endif	
}

void
create_file_tag(header)
	register union record *header;
{
	extern union record *head;		/* Points to current tape header */
	extern int head_standard;		/* Tape header is in ANSI format */
	extern FILE *msg_file;
	extern FILE *mapfile;
	extern long baserec;
	char buf[30];
	static int oldvol = -1;

#ifdef TAPE_IO
	if (f_map_mode == 'a' && !qfamap && _isrmt(archive)) {
		/*
		 * Get a new physical base address for every
		 * 1MB of data. Calling tape_tell too often
		 * slows down TAR. Make sure the treshold
		 * is a multiple of blocksize.
		 */
		static long prev_baserec;
		static long baseblk;
		static long treshold = 0;
		if (treshold == 0 || header->header.linkflag == 'V') {
			treshold = ((f_cluster * 1024L * 1024L) / blocksize) * blocking;
			prev_baserec = baserec;
			baseblk = rmtseek(archive, 0L, 4);
			oldvol = volno;
		} else if (baserec >= prev_baserec + treshold) {
			prev_baserec = baserec;
			baseblk = rmtseek(archive, 0L, 4);
		}
		if (baseblk >= 0)
			fprintf(mapfile, "%ld+%ld,", baseblk,
				(baserec - prev_baserec) + (ar_record - ar_block));
	}
#endif
	fprintf(mapfile, "%ld\t", baserec + (ar_record - ar_block));

	/* These globals are parameters to print_header, sigh */
	head_standard = f_standard;
	if (header->header.linkflag == 'V')
		print_header2(mapfile, header, header->header.name, ListLong|ListUnquoted);
	else
		print_header2(mapfile, header, filename, ListLong|ListUnquoted);

	if (ferror(mapfile)) {
		if (mapfile2) {
			msg("Error writing temporary tape directory, aborting QFA support");
			fclose(mapfile2);
		} else
			msg("Error writing tape directory, aborting QFA support");
		fclose(mapfile);
		f_map_file = 0;
		f_map_mode = 0;
	}
};

/*****************************************************************************/

	/* start writing new volume in buffered QFA mode */
void
dir_qfa_init(void)
{
	qfalen = 0;
}

	/* next record/id pair */
void
dir_qfa_record(long record, long block)
{
	if (qfalen >= qfasize) {
		qfasize = qfasize ? 2 * qfasize : 10 * 1024 * 2 * sizeof(long);
		qfamap = realloc(qfamap, qfasize);
		if (!qfamap) {
			msg("Out of memory (QFA map)");
			exit(EX_SYSTEM);
		}
	}
	qfamap[qfalen++] = record;
	qfamap[qfalen++] = block;
}

	/* terminate writing a volume in buffered QFA mode */
void
dir_qfa_term(void)
{
	char line[60 + FILENAME_MAX];
	int i = 0;

	fflush(mapfile);
	if (ferror(mapfile)) {
		msg("Error writing temporary tape directory, aborting QFA support");
		fclose(mapfile);
		fclose(mapfile2);
		f_map_file = 0;
		f_map_mode = 0;
		return;
	}

	/*
	 * Create final tape directory (mapfile2) by combining the temporary
	 * directory (mapfile) with record numbers only and the cluster block
	 * id map (qfamap).
	 */
	rewind(mapfile);
	while (fgets(line, 60 + FILENAME_MAX, mapfile)) {
		long blk, off, rec;
		char tag, *name;

		if ((name = decode_234(line, &blk, &off, &rec, &tag)) == NULL)
			continue;

		while (i+2 < qfalen && qfamap[i+2] <= rec)
			i += 2;
		fprintf(mapfile2, "%ld+%ld,%s",
			qfamap[i+1], rec - qfamap[i], line);
	}

	fflush(mapfile2);
	if (ferror(mapfile2)) {
		msg("Error writing tape directory, aborting QFA support");
		fclose(mapfile2);
		fclose(mapfile);
		f_map_file = 0;
		f_map_mode = 0;
	} else {
		fclose(mapfile);
		mapfile = fopen(mapname, "w+b");
	}
}

/*****************************************************************************/

void
dir_init(void)
{
	if (f_map_file) {
		switch(cmd_mode) {
		default:
			mapfile = fopen(f_map_file, "a");
			break;
		case CMD_EXTRACT:
		case CMD_DIFF:
			map_read = 1;
			mapfile = fopen(f_map_file, "r");
		}
		if (mapfile == NULL) {
			fprintf(stderr, "Cannot open map file %s\n", f_map_file);
			exit(EX_ARGSBAD);
		}

		if (f_map_mode == 'a' && f_buffered && !map_read) {
			mapfile2 = mapfile;
			mapname = tempnam(NULL, "tar");
			mapfile = fopen(mapname, "w+b");
			if (!mapfile) {
				msg_perror("Cannot create temporary file %s", mapname);
				exit(EX_SYSTEM);
			}
		}

		setvbuf(mapfile, NULL, _IOFBF, 16384);
	}
}

void
dir_term()
{
	if (f_map_file)
		fclose(mapfile);
	if (mapfile2)
		fclose(mapfile2);
	if (mapname)
		remove(mapname);
}
